#!/bin/bash -eu

SITE_PREFIX=$(gum input --placeholder "Site prefix - eg. leave empty for example.org, doc for doc.example.org")
SERVER_URL=$(gum input --placeholder "Server root URL - eg. example.org")
if [ -z "$SITE_PREFIX" ]; then
  SITE_NAME=www
  SITE_URL=$SERVER_URL
else
SITE_NAME=${SITE_PREFIX//./-}
SITE_URL=$SITE_PREFIX.$SERVER_URL
fi;

NS=${SERVER_URL/./-}
RESOURCE_NAME=$SITE_NAME--si
BUCKET_NAME=$NS-$RESOURCE_NAME

# mkdir -p /root/domains/$NS

NS_EXISTS=$(kubectl get namespace $NS --ignore-not-found);
if [[ "$NS_EXISTS" ]]; then
  echo "Skipping creation of namespace $NS - already exists";
else
  echo "Creating namespace $NS";
  kubectl create ns $NS;
fi;

kubectl apply -f - <<EOF
apiVersion: core.libre.sh/v1alpha1
kind: Bucket
metadata:
  labels:
    app.kubernetes.io/component: site
    app.kubernetes.io/instance: $SITE_NAME
    app.kubernetes.io/name: site
  name: $RESOURCE_NAME
  namespace: $NS
spec:
  provider: data
  policy:
    preset: cdn
EOF

kubectl apply -f - <<EOF
apiVersion: v1
kind: Service
metadata:
  labels:
    app.kubernetes.io/component: site
    app.kubernetes.io/instance: $SITE_NAME
    app.kubernetes.io/name: site
  name: $RESOURCE_NAME
  namespace: $NS
spec:
  type: ExternalName
  externalName: minio.minio-hot
EOF

kubectl apply -f - <<EOF
apiVersion: networking.k8s.io/v1
kind: Ingress
metadata:
  labels:
    app.kubernetes.io/component: site
    app.kubernetes.io/instance: $SITE_NAME
    app.kubernetes.io/name: site
  annotations:
    cert-manager.io/cluster-issuer: letsencrypt
    kubernetes.io/tls-acme: "true"
    nginx.ingress.kubernetes.io/backend-protocol: HTTPS
    nginx.ingress.kubernetes.io/configuration-snippet: |
      rewrite ^(.*)/$ \$1/index.html;
      rewrite ^(.*)$ /$BUCKET_NAME\$1 break;
    nginx.ingress.kubernetes.io/proxy-body-size: "0"
    nginx.ingress.kubernetes.io/proxy-ssl-verify: "off"
    nginx.ingress.kubernetes.io/rewrite-target: /$BUCKET_NAME/\$1
    nginx.ingress.kubernetes.io/server-snippet: |
      client_max_body_size 0;
  name: $RESOURCE_NAME
  namespace: $NS
spec:
  ingressClassName: nginx
  rules:
    - host: $SITE_URL
      http:
        paths:
        - backend:
            service:
              name: $RESOURCE_NAME
              port:
                number: 443
          path: /
          pathType: Prefix
  tls:
    - hosts:
        - $SITE_URL
      secretName: ${SITE_URL//./-}-tls
EOF
