package apps

import (
	"context"
	"fmt"
	"strings"
	"time"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/api/equality"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/util/rand"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/log"

	"github.com/Nerzal/gocloak/v11"
	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	"github.com/jinzhu/copier"
	"github.com/minio/minio-go/v7"

	appsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	"libre.sh/controller/internal"
)

// RealmReconciler reconciles a Realm object
type RealmReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=apps.libre.sh,resources=realms,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=apps.libre.sh,resources=realms/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=apps.libre.sh,resources=realms/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Realm object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *RealmReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)
	log.Info("reconciling")

	var realm appsv1alpha1.Realm
	err = r.Client.Get(ctx, req.NamespacedName, &realm)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if realm.Spec.Suspend {
		// TODO add suspend condition in status ?
		return ctrl.Result{}, nil
	}

	var kcErr error

	defer func() {
		if err == nil && minio.IsNetworkOrHostDown(kcErr, false) {
			err = kcErr
		}
		if err != nil {
			return
		}
		if kcErr != nil {
			log.Error(kcErr, "Failed to provision realm")
			result.RequeueAfter = 10 * time.Minute
			conditions.MarkStalled(&realm, meta.FailedReason, kcErr.Error())
		} else {
			conditions.Delete(&realm, meta.StalledCondition)
		}
		if conditions.IsStalled(&realm) {
			conditions.MarkFalse(&realm, meta.ReadyCondition, meta.FailedReason, "Is stalled")
		} else if conditions.IsReconciling(&realm) {
			conditions.MarkUnknown(&realm, meta.ReadyCondition, meta.ProgressingReason, "Is reconciling")
		} else {
			conditions.MarkTrue(&realm, meta.ReadyCondition, meta.SucceededReason, "Is provisionned")
		}
		err = r.Client.Status().Update(ctx, &realm)
	}()

	if realm.Generation != conditions.GetObservedGeneration(&realm, meta.ReadyCondition) {
		conditions.MarkReconciling(&realm, meta.ProgressingReason, "Reconciliation in progress")
		err = r.Client.Status().Update(ctx, &realm)
		if err != nil {
			return
		}
	}

	config, err := internal.GetConfig(ctx, r.Client, realm.Namespace)
	if err != nil {
		return
	}

	provider, err := config.GetKeycloak(ctx, r.Client)
	if err != nil {
		return
	}

	// TODO this only works for new keycloak version that do not use the auth subpath
	keycloakClient := gocloak.NewClient(provider.Endpoint, gocloak.SetAuthAdminRealms("admin/realms"), gocloak.SetAuthRealms("realms"))
	token, kcErr := keycloakClient.LoginAdmin(ctx, provider.Username, provider.Password, "master")
	if kcErr != nil {
		return
	}

	realmExists := true

	realmRepresentation, kcErr := keycloakClient.GetRealm(ctx, token.AccessToken, realm.Name)
	if kcErr != nil {
		apiError := kcErr.(*gocloak.APIError)
		if apiError.Code == 404 && apiError.Message == "404 Not Found: Realm not found." {
			realmExists = false
			realmRepresentation = &gocloak.RealmRepresentation{}
			kcErr = nil
		} else {
			return
		}
	}

	realm.SetDefaults()

	existing := &gocloak.RealmRepresentation{}
	copier.Copy(existing, realmRepresentation)

	realm.MutateRealmRepresentation(realmRepresentation)

	// TODO it is equal even if BrowserSecurityHeaders are different, it seems it does not check well the diff with *map[string]string
	isEqual := equality.Semantic.DeepEqual(existing, realmRepresentation)
	// TODO check diff and log it instead ?
	// diff := deep.Equal(existing, realmRepresentation)

	if !realmExists {
		// Create Realm
		log.Info("creating realm")
		_, kcErr = keycloakClient.CreateRealm(ctx, token.AccessToken, *realmRepresentation)
		if kcErr != nil {
			return
		}
	} else if !isEqual {
		log.Info("updating realm")
		kcErr = keycloakClient.UpdateRealm(ctx, token.AccessToken, *realmRepresentation)
		if kcErr != nil {
			return
		}

	}

	kcErr = CreateUserAdminRoleAndGroup(ctx, keycloakClient, token.AccessToken, realmRepresentation)
	if kcErr != nil {
		return
	}

	user := gocloak.User{
		Username:      gocloak.StringP("system"),
		Email:         gocloak.StringP("contact@indiehosters.net"),
		FirstName:     gocloak.StringP("IndieHosters"),
		Enabled:       gocloak.BoolP(true),
		EmailVerified: gocloak.BoolP(true),
		//	Groups:    &[]string{"admin"},
	}

	getUsersParams := gocloak.GetUsersParams{
		Username: user.Username,
		Exact:    gocloak.BoolP(true),
	}

	users, kcErr := keycloakClient.GetUsers(ctx, token.AccessToken, realm.Name, getUsersParams)
	if kcErr != nil {
		return
	}

	if len(users) > 1 {
		// TODO return error, should only return one user
		return
	}

	if len(users) == 0 {
		log.Info("creating system user")
		var userID string
		userID, kcErr = keycloakClient.CreateUser(ctx, token.AccessToken, realm.Name, user)
		if kcErr != nil {
			apiError := kcErr.(*gocloak.APIError)
			// ignore user already exists
			if apiError.Code != 409 {
				return
			}
			kcErr = nil
		}
		user.ID = gocloak.StringP(userID)
	} else {
		// TODO update user ??
		user.ID = users[0].ID
	}

	secret := &corev1.Secret{}
	secret.Name = fmt.Sprintf("%s-realm", realm.Name)
	secret.Namespace = realm.Namespace
	opResult, kcErr := r.CreateOrUpdateRealmSecret(ctx, keycloakClient, token.AccessToken, realm, secret)
	if kcErr != nil {
		return
	}

	if opResult != controllerutil.OperationResultNone {
		kcErr = keycloakClient.SetPassword(ctx, token.AccessToken, *user.ID, realm.Name, string(secret.Data["password"]), false)
		// TODO remove ?
		if kcErr != nil && !strings.Contains(kcErr.Error(), "must not be equal to any of") {
			return
		}

	}

	getClientsParams := gocloak.GetClientsParams{
		ClientID: gocloak.StringP("realm-management"),
	}
	clients, kcErr := keycloakClient.GetClients(ctx, token.AccessToken, realm.Name, getClientsParams)
	if kcErr != nil {
		return

	}

	if len(clients) != 1 {
		// todo return err
	}

	// var clientID *string
	var role *gocloak.Role
	clientID := clients[0].ID

	getUsersByRoleParams := gocloak.GetUsersByRoleParams{}
	users, kcErr = keycloakClient.GetUsersByClientRoleName(ctx, token.AccessToken, realm.Name, *clientID, "realm-admin", getUsersByRoleParams)

	var systemUserHasRole bool
	for _, user := range users {
		if user.Username != nil && *user.Username == "system" {
			systemUserHasRole = true
			break
		}
	}

	if !systemUserHasRole {
		roles, kcErr := keycloakClient.GetClientRoles(ctx, token.AccessToken, realm.Name, *clientID, gocloak.GetRoleParams{})
		if kcErr != nil {
			return
		}

		for _, ro := range roles {
			if *ro.Name == "realm-admin" {
				role = ro
				break
			}
		}

		kcErr = keycloakClient.AddClientRoleToUser(ctx, token.AccessToken, realm.Name, *clientID, *user.ID, []gocloak.Role{*role})
		if kcErr != nil {
			return
		}
	}

	kcErr = UpdateRoleListClientScopeProtocolMapper(ctx, keycloakClient, token.AccessToken, realmRepresentation)
	if kcErr != nil {
		return
	}
	conditions.Delete(&realm, meta.ReconcilingCondition)
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *RealmReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&appsv1alpha1.Realm{}).
		Complete(r)
}

var userAdminRolesName = []string{
	"manage-users",
	"query-users",
	"view-users",
	"query-groups",
}

func CreateUserAdminRoleAndGroup(ctx context.Context, c gocloak.GoCloak, token string, realm *gocloak.RealmRepresentation) error {
	role, err := c.GetRealmRole(ctx, token, *realm.Realm, "user-admin")
	if err != nil {
		if strings.Contains(err.Error(), "404 Not Found") {
			//	fmt.Printf("realm role user-admin does not exist, creating: %s\n", *realm.Realm)
			role = &gocloak.Role{
				Name:      gocloak.StringP("user-admin"),
				Composite: gocloak.BoolP(true),
			}

			_, err := c.CreateRealmRole(ctx, token, *realm.Realm, *role)
			if err != nil {
				return err
			}

			role, err = c.GetRealmRole(ctx, token, *realm.Realm, "user-admin")
			if err != nil {
				return err
			}

		} else {
			return err
		}
	}

	var clientID *string
	userAdminRoles := []gocloak.Role{}
	params := gocloak.GetClientsParams{
		ClientID: gocloak.StringP("realm-management"),
	}

	clients, err := c.GetClients(ctx, token, *realm.Realm, params)
	if err != nil {
		return err

	}

	if len(clients) != 1 {
		// todo return err
	}

	clientID = clients[0].ID

	for _, roleName := range userAdminRolesName {
		r, err := c.GetClientRole(ctx, token, *realm.Realm, *clientID, roleName)
		if err != nil {
			return err
		}

		userAdminRoles = append(userAdminRoles, *r)
	}

	err = c.AddRealmRoleComposite(ctx, token, *realm.Realm, *role.Name, userAdminRoles)
	if err != nil {
		return err
	}

	// 	fmt.Printf("creating realm admin group: %s\n", *realm.Realm)

	group := gocloak.Group{
		Name: gocloak.StringP("admin"),
	}

	groupParams := gocloak.GetGroupsParams{
		Search: group.Name,
	}

	groups, err := c.GetGroups(ctx, token, *realm.Realm, groupParams)
	if err != nil {
		return err
	}

	for _, g := range groups {
		if *g.Name == "admin" {
			group.ID = g.ID
		}
	}

	// group does not exist, create it
	if group.ID == nil {
		id, err := c.CreateGroup(ctx, token, *realm.Realm, group)
		if err != nil {
			return err
		}
		group.ID = &id
	}

	err = c.AddRealmRoleToGroup(ctx, token, *realm.Realm, *group.ID, []gocloak.Role{*role})
	if err != nil {
		return err
	}

	return nil
}

func (r *RealmReconciler) CreateOrUpdateRealmSecret(ctx context.Context, c gocloak.GoCloak, token string, realm appsv1alpha1.Realm, secret *corev1.Secret) (result controllerutil.OperationResult, err error) {
	keyStore, err := c.GetKeyStoreConfig(ctx, token, realm.Name)
	if err != nil {
		return result, err
	}
	var cert *string

	for _, k := range *keyStore.Key {
		if *k.Algorithm == "RS256" {
			cert = k.Certificate
		}
	}

	result, err = controllerutil.CreateOrUpdate(context.TODO(), r.Client, secret, func() error {
		password := string(secret.Data["password"])
		if password == "" {
			password = rand.String(32)
		}
		data := map[string][]byte{
			"tls.crt": []byte(*cert),
			// TODO should it be hardcoded ?
			"username": []byte("system"),
			"password": []byte(password),
		}

		secret.Data = data

		return controllerutil.SetOwnerReference(&realm, secret, r.Scheme)
	})
	if err != nil {
		return result, err
	}

	return result, nil
}

func UpdateRoleListClientScopeProtocolMapper(ctx context.Context, c gocloak.GoCloak, token string, realm *gocloak.RealmRepresentation) error {
	scopes, err := c.GetClientScopes(ctx, token, *realm.Realm)
	if err != nil {
		return err
	}

	for _, s := range scopes {
		if s.Name != nil && *s.Name == "role_list" {
			clientScope, err := c.GetClientScope(ctx, token, *realm.Realm, *s.ID)
			if err != nil {
				return err
			}

			for _, m := range *clientScope.ProtocolMappers {
				if *m.Name == "role list" {
					m.ProtocolMappersConfig.Single = gocloak.StringP("true")

					err := c.UpdateClientScopeProtocolMapper(ctx, token, *realm.Realm, *s.ID, m)
					if err != nil {
						return err
					}
				}
			}

		}
	}

	return nil
}
