package apps

import (
	"context"

	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	appsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"
)

// NextcloudReconciler reconciles a Nextcloud object
type NextcloudReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Nextcloud object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.11.0/pkg/reconcile
func (r *NextcloudReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	var nextcloud appsv1alpha1.Nextcloud
	err = r.Client.Get(ctx, req.NamespacedName, &nextcloud)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if nextcloud.Spec.Suspend {
		return ctrl.Result{}, nil
	}

	log.Info("Reconciling")

	defer internal.UpdateStatus(ctx, r.Client, &nextcloud, &err)
	// defer func() {
	// 	if err != nil {
	// 		return
	// 	}
	// 	conditions.MarkTrue(&nextcloud, meta.ReadyCondition, meta.SucceededReason, "Up and running")
	// 	conditions.SetSummary(&nextcloud, meta.ReadyCondition,
	// 		conditions.WithNegativePolarityConditions(
	// 			meta.ReconcilingCondition,
	// 			meta.StalledCondition,
	// 			corev1alpha1.PostgrestNotReadyCondition,
	// 			corev1alpha1.BucketNotReadyCondition,
	// 		),
	// 	)
	// 	err = r.Client.Status().Update(ctx, &nextcloud)
	// }()

	if nextcloud.Generation != conditions.GetObservedGeneration(&nextcloud, meta.ReadyCondition) {
		conditions.MarkReconciling(&nextcloud, meta.ProgressingReason, "Reconciling in progress")
		err = r.Client.Status().Update(ctx, &nextcloud)
		if err != nil {
			return
		}
	}

	var postgres corev1alpha1.Postgresql
	postgres.SetName(nextcloud.Name)
	postgres.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &postgres, func() error {
		postgres.Spec.Suspend = nextcloud.Spec.Suspend
		return controllerutil.SetControllerReference(&nextcloud, &postgres, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetPostgresNotReady(&nextcloud, &postgres)

	var bucket corev1alpha1.Bucket
	bucket.SetName(nextcloud.Name)
	bucket.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &bucket, func() error {
		bucket.Spec.Suspend = nextcloud.Spec.Suspend
		bucket.Spec.Provider = corev1alpha1.BucketDataProvider
		bucket.Spec.Type = corev1alpha1.BucketPrivateType
		return controllerutil.SetControllerReference(&nextcloud, &bucket, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetBucketNotReady(&nextcloud, &bucket)

	conditions.Delete(&nextcloud, meta.ReconcilingCondition)
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *NextcloudReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&appsv1alpha1.Nextcloud{}).
		Owns(&corev1alpha1.Postgresql{}).
		Owns(&corev1alpha1.Bucket{}).
		Watches(
			&source.Kind{Type: &appsv1alpha1.LibreConfig{}},
			handler.EnqueueRequestsFromMapFunc(r.findObjectsForLibreConfig),
			builder.WithPredicates(predicate.ResourceVersionChangedPredicate{}),
		).
		Complete(r)
}

func (r *NextcloudReconciler) findObjectsForLibreConfig(libreConf client.Object) []reconcile.Request {
	ncList := &appsv1alpha1.NextcloudList{}
	listOps := &client.ListOptions{}
	if libreConf.GetNamespace() != internal.GlobalNS {
		listOps.Namespace = libreConf.GetNamespace()
	}
	err := r.List(context.TODO(), ncList, listOps)
	if err != nil {
		return []reconcile.Request{}
	}

	requests := make([]reconcile.Request, len(ncList.Items))
	for i, item := range ncList.Items {
		requests[i] = reconcile.Request{
			NamespacedName: types.NamespacedName{
				Name:      item.GetName(),
				Namespace: item.GetNamespace(),
			},
		}
	}
	return requests
}
