package core

import (
	"context"
	"fmt"
	"strings"

	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	pgv1 "github.com/zalando/postgres-operator/pkg/apis/acid.zalan.do/v1"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/log"
)

// PostgresqlReconciler reconciles a Postgresql object
type PostgresqlReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=internal.libre.sh,resources=postgresqls,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=internal.libre.sh,resources=postgresqls/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=internal.libre.sh,resources=postgresqls/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Postgresql object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *PostgresqlReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	var postgres corev1alpha1.Postgresql
	err = r.Client.Get(ctx, req.NamespacedName, &postgres)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if postgres.Spec.Suspend {
		return ctrl.Result{}, nil
	}

	log.Info("Reconciling")

	defer internal.UpdateStatus(ctx, r.Client, &postgres, &err)

	if postgres.Generation != conditions.GetObservedGeneration(&postgres, meta.ReadyCondition) {
		conditions.MarkReconciling(&postgres, meta.ProgressingReason, "Reconciliation in progress")
		err = r.Client.Status().Update(ctx, &postgres)
		if err != nil {
			return
		}
	}

	var bucket corev1alpha1.Bucket
	bucket.Name = fmt.Sprintf("%s-backup", postgres.Name)
	bucket.SetNamespace(postgres.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &bucket, func() error {
		bucket.Spec.Suspend = postgres.Spec.Suspend
		bucket.Spec.Provider = corev1alpha1.BucketBackupProvider
		bucket.Spec.Type = corev1alpha1.BucketPrivateType
		return controllerutil.SetControllerReference(&postgres, &bucket, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetBucketNotReady(&postgres, &bucket)
	if !conditions.IsReady(&bucket) {
		return
	}

	pg := pgv1.Postgresql{
		ObjectMeta: metav1.ObjectMeta{
			Name:      postgres.GetZalandoName(),
			Namespace: postgres.GetNamespace(),
		},
	}
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &pg, func() error {
		pg.Spec.Users = make(map[string]pgv1.UserFlags)
		pg.Spec.Users[postgres.Name] = pgv1.UserFlags{
			"superuser",
			"createdb",
		}
		pg.Spec.Databases = make(map[string]string)
		pg.Spec.Databases[postgres.Name] = postgres.Name
		pg.Spec.TeamID = corev1alpha1.ZalandoTeam
		pg.Spec.NumberOfInstances = 2
		pg.Spec.PostgresqlParam = pgv1.PostgresqlParam{
			PgVersion: "14",
		}
		pg.Spec.Volume = pgv1.Volume{
			Size: "5G",
		}
		pg.Spec.Env = []corev1.EnvVar{
			{
				Name:  "AWS_S3_FORCE_PATH_STYLE",
				Value: "true",
			},
			{
				Name:  "WALG_DISABLE_S3_SSE",
				Value: "true",
			},
			{
				Name:  "USE_WALG_BACKUP",
				Value: "true",
			},
			{
				Name:  "USE_WALG_RESTORE",
				Value: "true",
			},
			{
				Name:  "CLONE_USE_WALG_RESTORE",
				Value: "true",
			},
			{
				Name:  "BACKUP_SCHEDULE",
				Value: "*/5 * * * *",
			},
			{
				Name:  "BACKUP_SCHEDULE",
				Value: "15",
			},
			{
				Name: "WAL_S3_BUCKET",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "bucket",
					},
				},
			},
			{
				Name: "AWS_ENDPOINT",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "url",
					},
				},
			},
			{
				Name: "AWS_ACCESS_KEY_ID",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "accessKey",
					},
				},
			},
			{
				Name: "AWS_SECRET_ACCESS_KEY",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "secretKey",
					},
				},
			},
		}
		return controllerutil.SetControllerReference(&postgres, &pg, r.Scheme)
	})
	if err != nil {
		return
	}

	switch pg.Status.PostgresClusterStatus {
	case pgv1.ClusterStatusAddFailed, pgv1.ClusterStatusSyncFailed, pgv1.ClusterStatusInvalid, pgv1.ClusterStatusUpdateFailed:
		conditions.MarkStalled(&postgres, meta.FailedReason, "Posgresql status is %s", pg.Status.PostgresClusterStatus)
	case pgv1.ClusterStatusRunning:
		conditions.Delete(&postgres, meta.StalledCondition)
		conditions.Delete(&postgres, meta.ReconcilingCondition)
	case "":
	default:
		conditions.MarkStalled(&postgres, meta.ProgressingReason, "Postgresql is %s", strings.ToLower(pg.Status.PostgresClusterStatus))
	}

	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *PostgresqlReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1alpha1.Postgresql{}).
		Owns(&pgv1.Postgresql{}).
		Owns(&corev1alpha1.Bucket{}).
		Complete(r)
}
