package apps

import (
	"context"
	"time"

	"github.com/fluxcd/pkg/runtime/conditions"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/log"

	appsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	"libre.sh/controller/pkg/core"
)

// NextcloudReconciler reconciles a Nextcloud object
type NextcloudReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Nextcloud object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.11.0/pkg/reconcile
func (r *NextcloudReconciler) Reconcile(ctx context.Context, req ctrl.Request) (ctrl.Result, error) {
	_ = log.FromContext(ctx)

	var nextcloud appsv1alpha1.Nextcloud
	err := r.Client.Get(ctx, req.NamespacedName, &nextcloud)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}
	defer func() {
		conditions.SetSummary(&nextcloud, "Ready")
		r.Client.Status().Update(ctx, &nextcloud)
	}()

	conditions.MarkFalse(&nextcloud, "PostgresReady", "PostgresReconciling", "Reconciling Postgres")
	postgres := core.Postgres{
		Name:      nextcloud.Name,
		Namespace: nextcloud.Namespace,
	}
	controllerRef := metav1.NewControllerRef(&nextcloud, nextcloud.GroupVersionKind())
	ready, err := postgres.Reconcile(ctx, r.Client, []metav1.OwnerReference{*controllerRef})
	if err != nil {
		return ctrl.Result{}, err
	}

	if !ready {
		return ctrl.Result{RequeueAfter: 10 * time.Second}, nil
	}
	conditions.MarkTrue(&nextcloud, "PostgresReady", "PostgresRunning", "Postgres is running")

	// TODO(user): your logic here

	return ctrl.Result{}, nil
}

// SetupWithManager sets up the controller with the Manager.
func (r *NextcloudReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&appsv1alpha1.Nextcloud{}).
		Complete(r)
}
