package v1alpha1

import (
	"fmt"

	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

type BucketType string

const (
	BucketCDNType     BucketType = "cdn"
	BucketPublicType  BucketType = "public"
	BucketPrivateType BucketType = "private"
)

type BucketProvider string

const (
	BucketDataProvider   BucketProvider = "data"
	BucketBackupProvider BucketProvider = "backup"
)

const BucketSecretSuffix = "bucket.libre.sh"

// BucketSpec defines the desired state of Bucket
type BucketSpec struct {
	// +optional
	Suspend bool `json:"suspend,omitempty"`
	// +required
	Type BucketType `json:"type,omitempty"`
	// +required
	Provider BucketProvider `json:"provider,omitempty"`
}

// BucketStatus defines the observed state of Bucket
type BucketStatus struct {
	//+optional
	Conditions []metav1.Condition `json:"conditions,omitempty"`
}

//+kubebuilder:object:root=true
//+kubebuilder:subresource:status
// +kubebuilder:printcolumn:name="Age",type="date",JSONPath=".metadata.creationTimestamp",description=""
// +kubebuilder:printcolumn:name="Ready",type="string",JSONPath=".status.conditions[?(@.type==\"Ready\")].status",description=""
// +kubebuilder:printcolumn:name="Status",type="string",JSONPath=".status.conditions[?(@.type==\"Ready\")].message",description=""

// Bucket is the Schema for the buckets API
type Bucket struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata,omitempty"`

	Spec   BucketSpec   `json:"spec,omitempty"`
	Status BucketStatus `json:"status,omitempty"`
}

//+kubebuilder:object:root=true

// BucketList contains a list of Bucket
type BucketList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata,omitempty"`
	Items           []Bucket `json:"items"`
}

func init() {
	SchemeBuilder.Register(&Bucket{}, &BucketList{})
}

func (o *Bucket) GetConditions() []metav1.Condition {
	return o.Status.Conditions
}

func (o *Bucket) SetConditions(conditions []metav1.Condition) {
	o.Status.Conditions = conditions
}

func (bucket *Bucket) GetBucketName() string {
	return fmt.Sprintf("%s-%s", bucket.Namespace, bucket.Name)
}

func (bucket *Bucket) GetBucketSecretName() string {
	return fmt.Sprintf("%s.%s", bucket.Name, BucketSecretSuffix)
}

func (o *Bucket) SecretName() string {
	return fmt.Sprintf("%s.bucket.libre.sh", o.Name)
}
