package v1alpha1

import (
	"fmt"

	"github.com/Nerzal/gocloak/v11"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// SAMLClientSpec defines the desired state of SAMLClient
type SAMLClientSpec struct {
	Suspend                        bool             `json:"suspend,omitempty"`
	ClientID                       string           `json:"clientID,omitempty"`
	Enabled                        *bool            `json:"enabled,omitempty"`
	RedirectURIs                   []string         `json:"redirectURIs,omitempty"`
	Realm                          string           `json:"realm,omitempty"`
	AssertionSignature             bool             `json:"assertionSignature,omitempty"`
	SigningCertificateSecretName   string           `json:"signingCertificateSecretName,omitempty"`
	SingleLogoutServiceURLRedirect string           `json:"singleLogoutServiceURLRedirect,omitempty"`
	SignatureAlgorithm             string           `json:"signatureAlgorithm,omitempty"`
	ClientSignature                bool             `json:"clientSignature,omitempty"`
	AuthnStatement                 bool             `json:"authnstatement,omitempty"`
	AssertionConsumerURLPost       string           `json:"assertionConsumerURLPost,omitempty"`
	SingleLogoutServiceURLPost     string           `json:"singleLogoutServiceURLPost,omitempty"`
	NameIDFormat                   string           `json:"nameIDFormat,omitempty"`
	ServerSignature                bool             `json:"serverSignature,omitempty"`
	ServerSignatureKeyInfoExt      bool             `json:"serverSignatureKeyInfoExt,omitempty"`
	FullScopeAllowed               bool             `json:"fullScopeAllowed,omitempty"`
	ProtocolMappers                []ProtocolMapper `json:"protocolMappers,omitempty"`
}

type ProtocolMapper struct {
	Config          map[string]string `json:"config,omitempty"`
	Name            string            `json:"name,omitempty"`
	Protocol        string            `json:"protocol,omitempty"`
	ProtocolMapper  string            `json:"protocolMapper,omitempty"`
	ConsentRequired bool              `json:"consentRequired,omitempty"`
}

// SAMLClientStatus defines the observed state of SAMLClient
type SAMLClientStatus struct {
	//+optional
	Conditions []metav1.Condition `json:"conditions,omitempty"`
}

//+kubebuilder:object:root=true
//+kubebuilder:subresource:status
//+kubebuilder:object:root=true
//+kubebuilder:subresource:status
// +kubebuilder:printcolumn:name="Age",type="date",JSONPath=".metadata.creationTimestamp",description=""
// +kubebuilder:printcolumn:name="Ready",type="string",JSONPath=".status.conditions[?(@.type==\"Ready\")].status",description=""
// +kubebuilder:printcolumn:name="Status",type="string",JSONPath=".status.conditions[?(@.type==\"Ready\")].message",description=""

// SAMLClient is the Schema for the samlclients API
type SAMLClient struct {
	metav1.TypeMeta   `json:",inline"`
	metav1.ObjectMeta `json:"metadata,omitempty"`

	Spec   SAMLClientSpec   `json:"spec,omitempty"`
	Status SAMLClientStatus `json:"status,omitempty"`
}

//+kubebuilder:object:root=true

// SAMLClientList contains a list of SAMLClient
type SAMLClientList struct {
	metav1.TypeMeta `json:",inline"`
	metav1.ListMeta `json:"metadata,omitempty"`
	Items           []SAMLClient `json:"items"`
}

func init() {
	SchemeBuilder.Register(&SAMLClient{}, &SAMLClientList{})
}

func (o *SAMLClient) GetConditions() []metav1.Condition {
	return o.Status.Conditions
}

func (o *SAMLClient) SetConditions(conditions []metav1.Condition) {
	o.Status.Conditions = conditions
}

func (o *SAMLClient) SetDefaults() {
	if o.Spec.Enabled == nil {
		o.Spec.Enabled = gocloak.BoolP(true)
	}
	if len(o.Spec.ClientID) == 0 {
		o.Spec.ClientID = o.Name
	}
}

func (o *SAMLClient) SecretName() string {
	return fmt.Sprintf("%s.saml.libre.sh", o.Name)
}
