package apps

import (
	"context"
	"fmt"
	"time"

	certmanager "github.com/cert-manager/cert-manager/pkg/apis/certmanager/v1"
	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	appsv1 "k8s.io/api/apps/v1"
	batchv1 "k8s.io/api/batch/v1"
	corev1 "k8s.io/api/core/v1"
	networkingv1 "k8s.io/api/networking/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/apimachinery/pkg/util/intstr"
	"k8s.io/apimachinery/pkg/util/rand"
	appsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"
)

const NextcloudImage = "libresh/nextcloud:23.0.8"
const NextcloudWebImage = "libresh/nextcloud:23.0.8-web"

var UserID int64 = 82

// NextcloudReconciler reconciles a Nextcloud object
type NextcloudReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=apps.libre.sh,resources=nextclouds/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Nextcloud object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.11.0/pkg/reconcile
func (r *NextcloudReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	var nextcloud appsv1alpha1.Nextcloud
	err = r.Client.Get(ctx, req.NamespacedName, &nextcloud)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if nextcloud.Spec.Suspend {
		return ctrl.Result{}, nil
	}

	log.Info("Reconciling")

	defer internal.UpdateStatus(ctx, r.Client, &nextcloud, &err)
	// defer func() {
	// 	if err != nil {
	// 		return
	// 	}
	// 	conditions.MarkTrue(&nextcloud, meta.ReadyCondition, meta.SucceededReason, "Up and running")
	// 	conditions.SetSummary(&nextcloud, meta.ReadyCondition,
	// 		conditions.WithNegativePolarityConditions(
	// 			meta.ReconcilingCondition,
	// 			meta.StalledCondition,
	// 			corev1alpha1.PostgrestNotReadyCondition,
	// 			corev1alpha1.BucketNotReadyCondition,
	// 		),
	// 	)
	// 	err = r.Client.Status().Update(ctx, &nextcloud)
	// }()

	if nextcloud.Generation != conditions.GetObservedGeneration(&nextcloud, meta.ReadyCondition) {
		conditions.MarkReconciling(&nextcloud, meta.ProgressingReason, "Reconciling in progress")
		err = r.Client.Status().Update(ctx, &nextcloud)
		if err != nil {
			return
		}
	}

	var postgres corev1alpha1.Postgres
	postgres.SetName(nextcloud.Name)
	postgres.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &postgres, func() error {
		postgres.Spec.Suspend = nextcloud.Spec.Suspend
		return controllerutil.SetControllerReference(&nextcloud, &postgres, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetPostgresNotReady(&nextcloud, &postgres)

	var redis corev1alpha1.Redis
	redis.SetName(nextcloud.Name)
	redis.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &redis, func() error {
		redis.Spec.Suspend = nextcloud.Spec.Suspend
		return controllerutil.SetControllerReference(&nextcloud, &redis, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetRedisNotReady(&nextcloud, &redis)

	var bucket corev1alpha1.Bucket
	bucket.SetName(nextcloud.Name)
	bucket.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &bucket, func() error {
		bucket.Spec.Suspend = nextcloud.Spec.Suspend
		bucket.Spec.Provider = corev1alpha1.BucketDataProvider
		bucket.Spec.Type = corev1alpha1.BucketPrivateType
		return controllerutil.SetControllerReference(&nextcloud, &bucket, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetBucketNotReady(&nextcloud, &bucket)

	// TODO manage certificate rotation
	var samlCertificate certmanager.Certificate
	samlCertificate.SetName(nextcloud.SAMLSecretName())
	samlCertificate.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &samlCertificate, func() error {
		samlCertificate.Spec.DNSNames = []string{nextcloud.Spec.Domain}
		samlCertificate.Spec.IssuerRef.Group = "cert-manager.io"
		samlCertificate.Spec.IssuerRef.Kind = "ClusterIssuer"
		samlCertificate.Spec.IssuerRef.Name = "selfsigned-issuer"
		samlCertificate.Spec.SecretName = nextcloud.SAMLSecretName()
		samlCertificate.Spec.CommonName = nextcloud.Spec.Domain

		// TODO
		d, _ := time.ParseDuration("26280h")
		samlCertificate.Spec.Duration = &metav1.Duration{Duration: d}
		// TODO
		subject := &certmanager.X509Subject{
			Organizations: []string{"indiehosters */"},
		}

		samlCertificate.Spec.Subject = subject

		return controllerutil.SetControllerReference(&nextcloud, &samlCertificate, r.Scheme)
	})
	if err != nil {
		return
	}

	var samlClient corev1alpha1.SAMLClient
	samlClient.SetName(nextcloud.Name)
	samlClient.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &samlClient, func() error {
		//	samlClient.Spec.Realm = "realm-sample"
		samlClient.Spec.Suspend = nextcloud.Spec.Suspend
		samlClient.Spec.ClientID = fmt.Sprintf("%s/apps/user_saml/saml/metadata", nextcloud.Spec.Domain)
		// samlClient.Spec.Enabled =
		samlClient.Spec.RedirectURIs = []string{fmt.Sprintf("%s/apps/user_saml/saml/acs", nextcloud.Spec.Domain)}
		samlClient.Spec.SigningCertificateSecretName = samlCertificate.Spec.SecretName
		samlClient.Spec.AssertionSignature = true
		samlClient.Spec.SingleLogoutServiceURLRedirect = fmt.Sprintf("%s/apps/user_saml/saml/sls", nextcloud.Spec.Domain)
		samlClient.Spec.SignatureAlgorithm = "RSA_SHA256"
		samlClient.Spec.ClientSignature = true
		samlClient.Spec.AuthnStatement = true
		samlClient.Spec.AssertionConsumerURLPost = fmt.Sprintf("%s/apps/user_saml/saml/acs", nextcloud.Spec.Domain)
		// TODO
		samlClient.Spec.SingleLogoutServiceURLPost = fmt.Sprintf("https://%s", nextcloud.Spec.Domain)
		samlClient.Spec.NameIDFormat = "username"
		samlClient.Spec.ServerSignature = true
		samlClient.Spec.ServerSignatureKeyInfoExt = false
		samlClient.Spec.FullScopeAllowed = true

		samlClient.Spec.ProtocolMappers = []corev1alpha1.ProtocolMapper{
			{
				Name:           "groups",
				Protocol:       "saml",
				ProtocolMapper: "saml-group-membership-mapper",
				Config: map[string]string{
					"single":               "true",
					"attribute.nameformat": "Basic",
					"full.path":            "false",
					"friendly.name":        "groups",
					"attribute.name":       "groups",
				},
			},
			{
				Name:           "email",
				Protocol:       "saml",
				ProtocolMapper: "saml-user-property-mapper",
				Config: map[string]string{
					"attribute.nameformat": "Basic",
					"full.path":            "false",
					"friendly.name":        "email",
					"user.attribute":       "email",
					"attribute.name":       "email",
				},
			},
			{
				Name:           "username",
				Protocol:       "saml",
				ProtocolMapper: "saml-user-property-mapper",
				Config: map[string]string{
					"attribute.nameformat": "Basic",
					"full.path":            "false",
					"friendly.name":        "username",
					"user.attribute":       "username",
					"attribute.name":       "username",
				},
			},
		}

		return controllerutil.SetControllerReference(&nextcloud, &samlClient, r.Scheme)
	})
	if err != nil {
		return
	}
	internal.SetSAMLNotReady(&nextcloud, &samlClient)

	var secretConf corev1.Secret
	secretConf.SetName(fmt.Sprintf("%s-config", nextcloud.Name))
	secretConf.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &secretConf, func() error {
		if secretConf.Data == nil {
			secretConf.Data = make(map[string][]byte)
		}
		if len(secretConf.Data["INSTANCE_ID"]) == 0 {
			instanceID := rand.String(7)
			secretConf.Data["INSTANCE_ID"] = []byte(instanceID)
		}

		if len(secretConf.Data["SECRET"]) == 0 {
			secret := rand.String(18)
			secretConf.Data["SECRET"] = []byte(secret)

		}

		if len(secretConf.Data["PASSWORD_SALT"]) == 0 {
			salt := rand.String(18)
			secretConf.Data["PASSWORD_SALT"] = []byte(salt)

		}

		if len(secretConf.Data["NEXTCLOUD_ADMIN_USER"]) == 0 {
			secretConf.Data["NEXTCLOUD_ADMIN_USER"] = []byte("admin")

		}

		if len(secretConf.Data["NEXTCLOUD_ADMIN_PASSWORD"]) == 0 {
			password := rand.String(32)
			secretConf.Data["NEXTCLOUD_ADMIN_PASSWORD"] = []byte(password)

		}

		return controllerutil.SetControllerReference(&nextcloud, &secretConf, r.Scheme)
	})
	if err != nil {
		return
	}

	var cm corev1.ConfigMap
	cm.SetName(fmt.Sprintf("%s-config", nextcloud.Name))
	cm.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &cm, func() error {
		data := map[string]string{
			"APPS_STORE_ENABLE":         "false",
			"CONFIG_READONLY":           "true",
			"DATA_DIRECTORY":            "/usr/src/nextcloud/data",
			"UPDATE_CHECKER":            "false",
			"UPDATE_DISABLE_WEB":        "true",
			"DISABLE_APPS":              "firstrunwizard",
			"ENABLE_APPS":               "user_saml,apporder,calendar,indie_external,admin_audit,activity",
			"TRUSTED_PROXIES":           "10.0.0.0/8",
			"PRIVACY_DATA_LOCATION":     "de",
			"OVERWRITE_CLI_URL":         fmt.Sprintf("https://%s", nextcloud.Spec.Domain),
			"NEXTCLOUD_TRUSTED_DOMAINS": nextcloud.Spec.Domain,

			"DB_TYPE": "pgsql",

			"OBJECTSTORE_S3_PORT":          "443",
			"OBJECTSTORE_S3_REGION":        "default",
			"OBJECTSTORE_S3_AUTOCREATE":    "false",
			"OBJECTSTORE_S3_USEPATH_STYLE": "true",
			"OBJECTSTORE_S3_SSL":           "true",

			"MAIL_DOMAIN": "liiib.re",
			"SMTP_HOST":   "mail.indie.host",
			"SMTP_PORT":   "587",
			"SMTP_SECURE": "tls",

			"SAML_IDP_URL":      "https://id.indie.host",
			"SAML_REALM":        "${NS}",
			"SAML_DISPLAY_NAME": "LiiibreSSO",

			// "CHAT_URL": "https://${CHAT_SUBDOMAIN}.${DOMAIN}",
		}
		cm.Data = data
		return controllerutil.SetControllerReference(&nextcloud, &cm, r.Scheme)
	})
	if err != nil {
		return
	}

	if conditions.IsTrue(&nextcloud, corev1alpha1.BucketNotReadyCondition) ||
		conditions.IsTrue(&nextcloud, corev1alpha1.PostgrestNotReadyCondition) ||
		conditions.IsTrue(&nextcloud, corev1alpha1.SAMLNotReadyCondition) ||
		conditions.IsTrue(&nextcloud, corev1alpha1.RedisNotReadyCondition) {
		return
	}

	var job batchv1.Job
	job.SetName(fmt.Sprintf("%s-install", nextcloud.Name))
	job.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &job, func() error {
		//	job.Spec.Template.Spec
		container := corev1.Container{
			Image:   NextcloudImage,
			Command: []string{"/install.sh"},
			Name:    "install",
			SecurityContext: &corev1.SecurityContext{
				RunAsUser:  &UserID,
				RunAsGroup: &UserID,
			},
			Env: r.getEnvVars(&nextcloud),
			EnvFrom: []corev1.EnvFromSource{
				{
					SecretRef: &corev1.SecretEnvSource{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: fmt.Sprintf("%s-config", nextcloud.Name),
						},
					},
				},
				{
					ConfigMapRef: &corev1.ConfigMapEnvSource{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: fmt.Sprintf("%s-config", nextcloud.Name),
						},
					},
				},
			},
		}

		job.Spec.Template.Spec.Containers = []corev1.Container{container}
		job.Spec.Template.Spec.RestartPolicy = corev1.RestartPolicyNever

		return controllerutil.SetControllerReference(&nextcloud, &job, r.Scheme)
	})
	if err != nil {
		return
	}
	if job.Status.CompletionTime == nil {
		conditions.MarkTrue(&nextcloud, "NotInstalled", "JobNotCompleted", "Waiting for install job to complete")
		return
	} else {
		conditions.Delete(&nextcloud, "NotInstalled")
	}

	var deployment appsv1.Deployment
	deployment.SetName(nextcloud.Name)
	deployment.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &deployment, func() error {
		deployment.Labels = nextcloud.Labels()
		// deployment.Spec.Replicas = 1
		deployment.Spec.Selector = &metav1.LabelSelector{
			MatchLabels: nextcloud.Labels(),
		}

		// TODO ?
		// deployment.Spec.Strategy

		deployment.Spec.Template.ObjectMeta = metav1.ObjectMeta{
			Labels: nextcloud.Labels(),
		}

		webPort := intstr.IntOrString{
			IntVal: 80,
		}

		appContainer := corev1.Container{
			Image: NextcloudImage,
			Name:  "app",
			Ports: []corev1.ContainerPort{
				{
					ContainerPort: 9000,
					Name:          "api",
					Protocol:      corev1.ProtocolTCP,
				},
			},
			ReadinessProbe: &corev1.Probe{
				ProbeHandler: corev1.ProbeHandler{
					Exec: &corev1.ExecAction{
						Command: []string{"/php-fpm-healthcheck"},
					},
				},
			},
			LivenessProbe: &corev1.Probe{
				ProbeHandler: corev1.ProbeHandler{
					HTTPGet: &corev1.HTTPGetAction{
						Path: "/heartbeat",
						Port: webPort,
						HTTPHeaders: []corev1.HTTPHeader{
							{
								Name:  "Host",
								Value: "localhost",
							},
						},
					},
				},
			},
			Command: []string{"/bin/sh"},
			Args:    []string{"-c", ">- /runtime/patches/apply.sh && php-fpm"},
			SecurityContext: &corev1.SecurityContext{
				RunAsUser:  &UserID,
				RunAsGroup: &UserID,
			},
			/*
				TODO
							resources:
							  requests:
								memory: "80Mi"
							  limits:
								memory: "400Mi"
			*/
		}

		envVars := r.getEnvVars(&nextcloud)
		envVars = append(envVars, corev1.EnvVar{
			Name:  "INSTALLED",
			Value: "true",
		})

		appContainer.Env = envVars

		webContainer := corev1.Container{
			Image: NextcloudWebImage,
			Name:  "web",
			Ports: []corev1.ContainerPort{
				{
					ContainerPort: 80,
					Name:          "http",
					Protocol:      corev1.ProtocolTCP,
				},
			},
			Env: []corev1.EnvVar{
				{
					Name:  "BACKEND_HOST",
					Value: "localhost",
				},
			},
			ReadinessProbe: &corev1.Probe{
				ProbeHandler: corev1.ProbeHandler{
					HTTPGet: &corev1.HTTPGetAction{
						Path: "/heartbeat",
						Port: webPort,
						HTTPHeaders: []corev1.HTTPHeader{
							{
								Name:  "Host",
								Value: "localhost",
							},
						},
					},
				},
			},
			LivenessProbe: &corev1.Probe{
				ProbeHandler: corev1.ProbeHandler{
					HTTPGet: &corev1.HTTPGetAction{
						Path: "/heartbeat",
						Port: webPort,
						HTTPHeaders: []corev1.HTTPHeader{
							{
								Name:  "Host",
								Value: "localhost",
							},
						},
					},
				},
			},
			/*
				TODO
					resources:
					  requests:
						memory: "80Mi"
					  limits:
						memory: "400Mi" */
		}

		deployment.Spec.Template.Spec.Containers = []corev1.Container{appContainer, webContainer}

		/*
			TODO
			   apiVersion: apps/v1
			   kind: Deployment
			   spec:
			     template:
			       spec:
			         affinity:
			           nodeAffinity:
			             preferredDuringSchedulingIgnoredDuringExecution:
			             - weight: 20
			               preference:
			                 matchExpressions:
			                 - key: stateless
			                   operator: In
			                   values:
			                   - "true"
		*/

		return controllerutil.SetControllerReference(&nextcloud, &deployment, r.Scheme)
	})
	if err != nil {
		return
	}

	var svc corev1.Service
	svc.SetName(nextcloud.Name)
	svc.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &svc, func() error {
		targetPort := intstr.IntOrString{
			StrVal: "http",
			Type:   intstr.String,
		}
		svc.Spec.Ports = []corev1.ServicePort{
			{
				Name:       "http",
				Port:       80,
				Protocol:   corev1.ProtocolTCP,
				TargetPort: targetPort,
			},
		}
		svc.Spec.Type = corev1.ServiceTypeClusterIP
		svc.Spec.Selector = nextcloud.Labels()

		return controllerutil.SetControllerReference(&nextcloud, &svc, r.Scheme)
	})
	if err != nil {
		return
	}

	var ingress networkingv1.Ingress
	ingress.SetName(nextcloud.Name)
	ingress.SetNamespace(nextcloud.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &ingress, func() error {
		ingress.ObjectMeta.Annotations = map[string]string{
			"kubernetes.io/tls-acme":                              "true",
			"nginx.ingress.kubernetes.io/proxy-body-size":         "100g",
			"nginx.ingress.kubernetes.io/proxy-request-buffering": "off",
		}

		pathType := networkingv1.PathTypePrefix

		ingress.Spec.Rules = []networkingv1.IngressRule{
			{
				Host: nextcloud.Spec.Domain,
				IngressRuleValue: networkingv1.IngressRuleValue{
					HTTP: &networkingv1.HTTPIngressRuleValue{
						Paths: []networkingv1.HTTPIngressPath{
							{
								Path:     "/",
								PathType: &pathType,
								Backend: networkingv1.IngressBackend{
									Service: &networkingv1.IngressServiceBackend{
										Name: nextcloud.Name,
										Port: networkingv1.ServiceBackendPort{
											Name: "http",
										},
									},
								},
							},
						},
					},
				},
			},
		}

		ingress.Spec.TLS = []networkingv1.IngressTLS{
			{
				Hosts:      []string{nextcloud.Spec.Domain},
				SecretName: fmt.Sprintf("%s-tls", nextcloud.Name),
			},
		}

		return controllerutil.SetControllerReference(&nextcloud, &ingress, r.Scheme)
	})
	if err != nil {
		return
	}

	conditions.Delete(&nextcloud, meta.ReconcilingCondition)

	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *NextcloudReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&appsv1alpha1.Nextcloud{}).
		Owns(&corev1alpha1.Postgres{}).
		Owns(&corev1alpha1.Bucket{}).
		Owns(&corev1alpha1.SAMLClient{}).
		Owns(&batchv1.Job{}).
		Watches(
			&source.Kind{Type: &appsv1alpha1.LibreConfig{}},
			handler.EnqueueRequestsFromMapFunc(r.findObjectsForLibreConfig),
			builder.WithPredicates(predicate.ResourceVersionChangedPredicate{}),
		).
		Complete(r)
}

func (r *NextcloudReconciler) findObjectsForLibreConfig(libreConf client.Object) []reconcile.Request {
	ncList := &appsv1alpha1.NextcloudList{}
	listOps := &client.ListOptions{}
	if libreConf.GetNamespace() != internal.GlobalNS {
		listOps.Namespace = libreConf.GetNamespace()
	}
	err := r.List(context.TODO(), ncList, listOps)
	if err != nil {
		return []reconcile.Request{}
	}

	requests := make([]reconcile.Request, len(ncList.Items))
	for i, item := range ncList.Items {
		requests[i] = reconcile.Request{
			NamespacedName: types.NamespacedName{
				Name:      item.GetName(),
				Namespace: item.GetNamespace(),
			},
		}
	}
	return requests
}

func (r *NextcloudReconciler) getEnvVars(nextcloud *appsv1alpha1.Nextcloud) []corev1.EnvVar {
	bucket := fmt.Sprintf("%s.bucket.libre.sh", nextcloud.Name)
	postgres := fmt.Sprintf("%s.postgres.libre.sh", nextcloud.Name)
	redis := fmt.Sprintf("%s.redis.libre.sh", nextcloud.Name)
	samlclient := fmt.Sprintf("%s.samlclient.libre.sh", nextcloud.Name)
	return []corev1.EnvVar{
		{
			Name: "OBJECTSTORE_S3_KEY",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: bucket,
					},
					Key: "accessKey",
				},
			},
		},
		{
			Name: "OBJECTSTORE_S3_SECRET",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: bucket,
					},
					Key: "secretKey",
				},
			},
		},
		{
			Name: "OBJECTSTORE_S3_HOST",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: bucket,
					},
					Key: "endpoint",
				},
			},
		},
		{
			Name: "OBJECTSTORE_S3_BUCKET",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: bucket,
					},
					Key: "bucket",
				},
			},
		},
		// {
		// 	Name: "MAIL_FROM_ADDRESS",
		// 	ValueFrom: &corev1.EnvVarSource{
		// 		SecretKeyRef: &corev1.SecretKeySelector{
		// 			LocalObjectReference: corev1.LocalObjectReference{
		// 				Name: "",
		// 			},
		// 			Key: "",
		// 		},
		// 	},
		// },
		// {
		// 	Name: "SMTP_PASSWORD",
		// 	ValueFrom: &corev1.EnvVarSource{
		// 		SecretKeyRef: &corev1.SecretKeySelector{
		// 			LocalObjectReference: corev1.LocalObjectReference{
		// 				Name: "",
		// 			},
		// 			Key: "",
		// 		},
		// 	},
		// },
		// {
		// 	Name: "SMTP_NAME",
		// 	ValueFrom: &corev1.EnvVarSource{
		// 		SecretKeyRef: &corev1.SecretKeySelector{
		// 			LocalObjectReference: corev1.LocalObjectReference{
		// 				Name: "",
		// 			},
		// 			Key: "",
		// 		},
		// 	},
		// },
		{
			Name: "DB_USER",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: postgres,
					},
					Key: "username",
				},
			},
		},
		{
			Name: "DB_PASSWORD",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: postgres,
					},
					Key: "password",
				},
			},
		},
		{
			Name: "DB_HOST",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: postgres,
					},
					Key: "endpoint",
				},
			},
		},
		{
			Name: "DB_PORT",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: postgres,
					},
					Key: "port",
				},
			},
		},
		{
			Name: "DB_NAME",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: postgres,
					},
					Key: "database",
				},
			},
		},

		{
			Name: "REDIS_HOST",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: redis,
					},
					Key: "endpoint",
				},
			},
		},
		{
			Name: "REDIS_HOST_PORT",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: redis,
					},
					Key: "port",
				},
			},
		},
		{
			Name: "REDIS_PORT",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: redis,
					},
					Key: "port",
				},
			},
		},
		{
			Name: "SAML_CERT",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: samlclient,
					},
					Key: "tls.crt",
				},
			},
		},
		{
			Name: "SAML_KEY",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: samlclient,
					},
					Key: "tls.key",
				},
			},
		},
		{
			Name: "SAML_IDP_CERT",
			ValueFrom: &corev1.EnvVarSource{
				SecretKeyRef: &corev1.SecretKeySelector{
					LocalObjectReference: corev1.LocalObjectReference{
						Name: samlclient,
					},
					Key: "tls.crt",
				},
			},
		},
	}
}
