package apps

import (
	"context"
	"strings"
	"time"

	corev1 "k8s.io/api/core/v1"
	apierrors "k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/apimachinery/pkg/util/rand"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/log"

	"github.com/Nerzal/gocloak/v11"
	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"

	appsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	"libre.sh/controller/internal"
)

// RealmReconciler reconciles a Realm object
type RealmReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=apps.libre.sh,resources=realms,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=apps.libre.sh,resources=realms/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=apps.libre.sh,resources=realms/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Realm object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *RealmReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)
	log.Info("reconciling")

	var realm appsv1alpha1.Realm
	err = r.Client.Get(ctx, req.NamespacedName, &realm)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if realm.Spec.Suspend {
		// TODO add suspend condition in status ?
		return ctrl.Result{}, nil
	}

	var kcErr error

	defer func() {
		if err != nil {
			return
		}
		if kcErr != nil {
			log.Error(kcErr, "Failed to provision realm")
			result.RequeueAfter = 10 * time.Minute
			conditions.MarkStalled(&realm, meta.FailedReason, kcErr.Error())
		} else {
			conditions.Delete(&realm, meta.StalledCondition)
		}
		if conditions.IsStalled(&realm) {
			conditions.MarkFalse(&realm, meta.ReadyCondition, meta.FailedReason, "Is stalled")
		} else if conditions.IsReconciling(&realm) {
			conditions.MarkUnknown(&realm, meta.ReadyCondition, meta.ProgressingReason, "Is reconciling")
		} else {
			conditions.MarkTrue(&realm, meta.ReadyCondition, meta.SucceededReason, "Is provisionned")
		}
		err = r.Client.Status().Update(ctx, &realm)
	}()

	if realm.Generation != conditions.GetObservedGeneration(&realm, meta.ReadyCondition) {
		conditions.MarkReconciling(&realm, meta.ProgressingReason, "Reconciliation in progress")
		err = r.Client.Status().Update(ctx, &realm)
		if err != nil {
			return
		}
	}

	config, err := internal.GetConfig(ctx, r.Client, realm.Namespace)
	if err != nil {
		return
	}

	provider, err := config.GetKeycloak(ctx, r.Client)
	if err != nil {
		return
	}

	systemUserSecretExists, systemUserCanConnect := true, true
	var keycloakClient gocloak.GoCloak
	var token *gocloak.JWT
	secret := &corev1.Secret{}

	err = r.Client.Get(ctx, types.NamespacedName{Namespace: realm.Namespace, Name: realm.GetUserSecretName()}, secret)
	if err != nil {
		if apierrors.IsNotFound(err) {
			systemUserSecretExists = false
			err = nil
		} else {
			return
		}
	}

	// TODO this only works for new keycloak version that do not use the auth subpath
	if systemUserSecretExists {
		keycloakClient = gocloak.NewClient(provider.Endpoint, gocloak.SetAuthAdminRealms("admin/realms"), gocloak.SetAuthRealms("realms"))
		token, kcErr = keycloakClient.LoginAdmin(ctx, string(secret.Data["username"]), string(secret.Data["password"]), realm.Namespace)
		if kcErr != nil {
			apiError := kcErr.(*gocloak.APIError)
			if apiError.Code != 401 && apiError.Message != "401 Unauthorized: invalid_grant: Invalid user credentials" {
				return
			}
			systemUserCanConnect = false
			kcErr = nil
		} else {
			systemUserCanConnect = true
		}
	}

	if !systemUserCanConnect || !systemUserSecretExists {
		// TODO this only works for new keycloak version that do not use the auth subpath
		keycloakClient = gocloak.NewClient(provider.Endpoint, gocloak.SetAuthAdminRealms("admin/realms"), gocloak.SetAuthRealms("realms"))
		token, kcErr = keycloakClient.LoginAdmin(ctx, provider.Username, provider.Password, "master")
		if kcErr != nil {
			return
		}
	}

	realm.SetDefaults()

	realmExists := true

	realmRepresentation, kcErr := keycloakClient.GetRealm(ctx, token.AccessToken, realm.Namespace)
	if kcErr != nil {
		apiError := kcErr.(*gocloak.APIError)
		if apiError.Code == 404 && apiError.Message == "404 Not Found: Realm not found." {
			realmExists = false
			realmRepresentation = &gocloak.RealmRepresentation{}
			kcErr = nil
		} else {
			return
		}
	}

	realm.SetDefaults()

	realm.MutateRealmRepresentation(realmRepresentation)

	if !realmExists {
		// Create Realm
		log.Info("creating realm")
		_, kcErr = keycloakClient.CreateRealm(ctx, token.AccessToken, *realmRepresentation)
		if kcErr != nil {
			return
		}
	} else {
		log.Info("updating realm")
		kcErr = keycloakClient.UpdateRealm(ctx, token.AccessToken, *realmRepresentation)
		if kcErr != nil {
			return
		}
	}

	if !systemUserCanConnect || !systemUserSecretExists {
		// TODO change token for the one of system user
		// TODO change secret name
		err = r.CreateOrUpdateSystemUser(ctx, keycloakClient, token.AccessToken, realm)
		if err != nil {
			return
		}
	}

	// TODO createOrUpdate secret with realm key

	kcErr = CreateUserAdminRoleAndGroup(ctx, keycloakClient, token.AccessToken, realmRepresentation)
	if kcErr != nil {
		return
	}

	kcErr = UpdateRoleListClientScopeProtocolMapper(ctx, keycloakClient, token.AccessToken, realmRepresentation)
	if kcErr != nil {
		return
	}

	kcErr = CreateOrUpdateGroupsClientScope(ctx, keycloakClient, token.AccessToken, realmRepresentation)
	if kcErr != nil {
		return
	}

	conditions.Delete(&realm, meta.ReconcilingCondition)
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *RealmReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&appsv1alpha1.Realm{}).
		Complete(r)
}

var userAdminRolesName = []string{
	"manage-users",
	"query-users",
	"view-users",
	"query-groups",
}

func CreateUserAdminRoleAndGroup(ctx context.Context, c gocloak.GoCloak, token string, realm *gocloak.RealmRepresentation) error {
	role, err := c.GetRealmRole(ctx, token, *realm.Realm, "user-admin")
	if err != nil {
		if strings.Contains(err.Error(), "404 Not Found") {
			role = &gocloak.Role{
				Name:      gocloak.StringP("user-admin"),
				Composite: gocloak.BoolP(true),
			}

			_, err := c.CreateRealmRole(ctx, token, *realm.Realm, *role)
			if err != nil {
				return err
			}

			role, err = c.GetRealmRole(ctx, token, *realm.Realm, "user-admin")
			if err != nil {
				return err
			}

		} else {
			return err
		}
	}

	var clientID *string
	userAdminRoles := []gocloak.Role{}
	params := gocloak.GetClientsParams{
		ClientID: gocloak.StringP("realm-management"),
	}

	clients, err := c.GetClients(ctx, token, *realm.Realm, params)
	if err != nil {
		return err

	}

	if len(clients) != 1 {
		// todo return err
	}

	clientID = clients[0].ID

	for _, roleName := range userAdminRolesName {
		r, err := c.GetClientRole(ctx, token, *realm.Realm, *clientID, roleName)
		if err != nil {
			return err
		}

		userAdminRoles = append(userAdminRoles, *r)
	}

	err = c.AddRealmRoleComposite(ctx, token, *realm.Realm, *role.Name, userAdminRoles)
	if err != nil {
		return err
	}

	group := gocloak.Group{
		Name: gocloak.StringP("admin"),
	}

	groupParams := gocloak.GetGroupsParams{
		Search: group.Name,
	}

	groups, err := c.GetGroups(ctx, token, *realm.Realm, groupParams)
	if err != nil {
		return err
	}

	for _, g := range groups {
		if *g.Name == "admin" {
			group.ID = g.ID
		}
	}

	// group does not exist, create it
	if group.ID == nil {
		id, err := c.CreateGroup(ctx, token, *realm.Realm, group)
		if err != nil {
			return err
		}
		group.ID = &id
	}

	err = c.AddRealmRoleToGroup(ctx, token, *realm.Realm, *group.ID, []gocloak.Role{*role})
	if err != nil {
		return err
	}

	return nil
}

func UpdateRoleListClientScopeProtocolMapper(ctx context.Context, c gocloak.GoCloak, token string, realm *gocloak.RealmRepresentation) error {
	scopes, err := c.GetClientScopes(ctx, token, *realm.Realm)
	if err != nil {
		return err
	}

	for _, s := range scopes {
		if s.Name != nil && *s.Name == "role_list" {
			clientScope, err := c.GetClientScope(ctx, token, *realm.Realm, *s.ID)
			if err != nil {
				return err
			}

			for _, m := range *clientScope.ProtocolMappers {
				if *m.Name == "role list" {
					m.ProtocolMappersConfig.Single = gocloak.StringP("true")

					err := c.UpdateClientScopeProtocolMapper(ctx, token, *realm.Realm, *s.ID, m)
					if err != nil {
						return err
					}
				}
			}

		}
	}

	return nil
}

func CreateOrUpdateGroupsClientScope(ctx context.Context, c gocloak.GoCloak, token string, realm *gocloak.RealmRepresentation) error {
	scopes, err := c.GetClientScopes(ctx, token, *realm.Realm)
	if err != nil {
		return err
	}

	clientScope := &gocloak.ClientScope{}

	for _, s := range scopes {
		if s.Name != nil && *s.Name == "groups" {
			clientScope, err = c.GetClientScope(ctx, token, *realm.Realm, *s.ID)
			if err != nil {
				return err
			}
			break
		}
	}

	clientScope.Name = gocloak.StringP("groups")
	clientScope.Protocol = gocloak.StringP("openid-connect")
	clientScope.Type = gocloak.StringP("default")
	clientScope.ClientScopeAttributes = &gocloak.ClientScopeAttributes{
		DisplayOnConsentScreen: gocloak.StringP("true"),
		IncludeInTokenScope:    gocloak.StringP("true"),
	}

	clientScopeProtocolMapper := gocloak.ProtocolMappers{
		Protocol:       gocloak.StringP("openid-connect"),
		ProtocolMapper: gocloak.StringP("oidc-group-membership-mapper"),
		Name:           gocloak.StringP("groups"),
		ProtocolMappersConfig: &gocloak.ProtocolMappersConfig{
			ClaimName:          gocloak.StringP("groups"),
			FullPath:           gocloak.StringP("false"),
			IDTokenClaim:       gocloak.StringP("true"),
			AccessTokenClaim:   gocloak.StringP("true"),
			UserinfoTokenClaim: gocloak.StringP("true"),
		},
	}

	clientScopeProtocolMappers := &[]gocloak.ProtocolMappers{clientScopeProtocolMapper}

	clientScope.ProtocolMappers = clientScopeProtocolMappers
	if clientScope.ID == nil {
		clientScopeID, err := c.CreateClientScope(ctx, token, *realm.Realm, *clientScope)
		if err != nil {
			return err
		}
		clientScope.ID = &clientScopeID
	} else {
		err := c.UpdateClientScope(ctx, token, *realm.Realm, *clientScope)
		if err != nil {
			return err
		}

		protocolMappers, err := c.GetClientScopeProtocolMappers(ctx, token, *realm.Realm, *clientScope.ID)
		if err != nil {
			return err
		}

		var protocolMapperExists bool
		for _, protocolMapper := range protocolMappers {
			if protocolMapper.Name != nil && *protocolMapper.Name == "groups" {
				clientScopeProtocolMapper.ID = protocolMapper.ID
			}
			protocolMapperExists = true
		}

		if !protocolMapperExists {
			id, err := c.CreateClientScopeProtocolMapper(ctx, token, *realm.Realm, *clientScope.ID, clientScopeProtocolMapper)
			if err != nil {
				return err
			}
			clientScopeProtocolMapper.ID = &id
		} else {
			err := c.UpdateClientScopeProtocolMapper(ctx, token, *realm.Realm, *clientScope.ID, clientScopeProtocolMapper)
			if err != nil {
				return err
			}
		}
	}

	return nil
}

func (r *RealmReconciler) CreateOrUpdateSystemUser(ctx context.Context, keycloakClient gocloak.GoCloak, token string, realm appsv1alpha1.Realm) error {
	user := gocloak.User{
		Username:      gocloak.StringP("system"),
		Email:         gocloak.StringP("contact@indiehosters.net"),
		FirstName:     gocloak.StringP("IndieHosters"),
		Enabled:       gocloak.BoolP(true),
		EmailVerified: gocloak.BoolP(true),
		//	Groups:    &[]string{"admin"},
	}

	getUsersParams := gocloak.GetUsersParams{
		Username: user.Username,
		Exact:    gocloak.BoolP(true),
	}

	users, kcErr := keycloakClient.GetUsers(ctx, token, realm.Namespace, getUsersParams)
	if kcErr != nil {
		return kcErr
	}

	if len(users) > 1 {
		// TODO return error, should only return one user
		return kcErr
	}

	if len(users) == 0 {
		// TODO add logger to realm reconciler
		// log.Info("creating system user")
		var userID string
		userID, kcErr = keycloakClient.CreateUser(ctx, token, realm.Namespace, user)
		if kcErr != nil {
			return kcErr
		}
		user.ID = gocloak.StringP(userID)
	} else {
		// TODO update user ??
		// Compare existing and new user, update if diff
		user.ID = users[0].ID
	}

	var secret corev1.Secret
	secret.SetName(realm.GetUserSecretName())
	secret.SetNamespace(realm.Namespace)
	_, err := controllerutil.CreateOrUpdate(context.TODO(), r.Client, &secret, func() error {
		password := string(secret.Data["password"])
		if password == "" {
			password = rand.String(32)
		}

		kcErr = keycloakClient.SetPassword(ctx, token, *user.ID, realm.Namespace, password, false)
		// TODO remove ?
		if kcErr != nil && !strings.Contains(kcErr.Error(), "must not be equal to any of") {
			return kcErr
		} else {
			password = rand.String(32)
			kcErr = keycloakClient.SetPassword(ctx, token, *user.ID, realm.Namespace, password, false)
			if kcErr != nil {
				return kcErr
			}
		}

		data := map[string][]byte{
			// TODO should it be hardcoded ?
			"username": []byte("system"),
			"password": []byte(password),
		}

		secret.Data = data

		return controllerutil.SetOwnerReference(&realm, &secret, r.Scheme)
	})
	if err != nil {
		return err
	}

	getClientsParams := gocloak.GetClientsParams{
		ClientID: gocloak.StringP("realm-management"),
	}
	clients, kcErr := keycloakClient.GetClients(ctx, token, realm.Namespace, getClientsParams)
	if kcErr != nil {
		return kcErr

	}

	if len(clients) != 1 {
		// todo return err
	}

	clientID := clients[0].ID

	getUsersByRoleParams := gocloak.GetUsersByRoleParams{}
	users, kcErr = keycloakClient.GetUsersByClientRoleName(ctx, token, realm.Namespace, *clientID, "realm-admin", getUsersByRoleParams)

	var systemUserHasRole bool
	for _, user := range users {
		if user.Username != nil && *user.Username == "system" {
			systemUserHasRole = true
			break
		}
	}

	if !systemUserHasRole {
		var role *gocloak.Role
		roles, kcErr := keycloakClient.GetClientRoles(ctx, token, realm.Namespace, *clientID, gocloak.GetRoleParams{})
		if kcErr != nil {
			return kcErr
		}

		for _, ro := range roles {
			if *ro.Name == "realm-admin" {
				role = ro
				break
			}
		}

		kcErr = keycloakClient.AddClientRoleToUser(ctx, token, realm.Namespace, *clientID, *user.ID, []gocloak.Role{*role})
		if kcErr != nil {
			return kcErr
		}
	}

	return nil
}
