package core

import (
	"context"
	"fmt"
	"time"

	"github.com/Nerzal/gocloak/v11"
	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	corev1 "k8s.io/api/core/v1"
	apierrors "k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/fields"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"

	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
)

// OIDCClientReconciler reconciles a OIDCClient object
type OIDCClientReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=core.libre.sh,resources=oidcclients,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=core.libre.sh,resources=oidcclients/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=core.libre.sh,resources=oidcclients/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the OIDCClient object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *OIDCClientReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	log.Info("reconciling")

	var oidcClient corev1alpha1.OIDCClient
	err = r.Client.Get(ctx, req.NamespacedName, &oidcClient)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if oidcClient.Spec.Suspend {
		// TODO set suspendended in status ?
		return ctrl.Result{}, nil
	}

	var kcErr error

	defer func() {
		if err != nil {
			return
		}
		if kcErr != nil {
			log.Error(kcErr, "Failed to provision oidc client")
			result.RequeueAfter = 10 * time.Minute
			conditions.MarkStalled(&oidcClient, meta.FailedReason, kcErr.Error())
		}
		if conditions.IsStalled(&oidcClient) {
			conditions.MarkFalse(&oidcClient, meta.ReadyCondition, meta.FailedReason, conditions.GetMessage(&oidcClient, meta.StalledCondition))
		} else if conditions.IsReconciling(&oidcClient) {
			conditions.MarkUnknown(&oidcClient, meta.ReadyCondition, meta.ProgressingReason, "Is reconciling")
		} else {
			conditions.MarkTrue(&oidcClient, meta.ReadyCondition, meta.SucceededReason, "Is provisionned")
		}
		err = r.Client.Status().Update(ctx, &oidcClient)
	}()

	if oidcClient.Generation != conditions.GetObservedGeneration(&oidcClient, meta.ReadyCondition) {
		conditions.MarkReconciling(&oidcClient, meta.ProgressingReason, "Reconciliation in progress")
		err = r.Client.Status().Update(ctx, &oidcClient)
		if err != nil {
			return
		}
	}

	secret := &corev1.Secret{}

	err = r.Client.Get(ctx, types.NamespacedName{Namespace: oidcClient.Namespace, Name: fmt.Sprintf("%s-realm", oidcClient.Spec.Realm)}, secret)
	if err != nil {
		if apierrors.IsNotFound(err) {
			conditions.MarkStalled(&oidcClient, meta.FailedReason, "Dependency is not ready")
			err = nil
		}
		return
	}

	config, err := internal.GetConfig(ctx, r.Client, oidcClient.Namespace)
	if err != nil {
		return
	}

	provider, err := config.GetKeycloak(ctx, r.Client)
	if err != nil {
		return
	}

	oidcClient.SetDefaults()

	// TODO this only works for new keycloak version that do not use the auth subpath
	// TODO use system user not master
	keycloakClient := gocloak.NewClient(provider.Endpoint, gocloak.SetAuthAdminRealms("admin/realms"), gocloak.SetAuthRealms("realms"))
	token, kcErr := keycloakClient.LoginAdmin(ctx, string(secret.Data["username"]), string(secret.Data["password"]), oidcClient.Spec.Realm)
	if kcErr != nil {
		return
	}

	getClientsParams := gocloak.GetClientsParams{
		ClientID: &oidcClient.Spec.ClientID,
	}

	var oidClientExists bool
	clientRepresentation := &gocloak.Client{}

	clientRepresentations, kcErr := keycloakClient.GetClients(ctx, token.AccessToken, oidcClient.Spec.Realm, getClientsParams)
	if kcErr != nil {
		return
	}

	// TODO manage it if returns multiple clients ?
	if len(clientRepresentations) == 1 {
		oidClientExists = true
		clientRepresentation = clientRepresentations[0]
	}

	internal.MutateOIDCClientRepresentation(oidcClient, clientRepresentation)

	if !oidClientExists {
		// Create Realm
		log.Info("creating oidc client")
		clientID, kcErr := keycloakClient.CreateClient(ctx, token.AccessToken, oidcClient.Spec.Realm, *clientRepresentation)
		if kcErr != nil {
			return
		}
		clientRepresentation.ID = &clientID
	} else {
		log.Info("updating oidc client")
		kcErr = keycloakClient.UpdateClient(ctx, token.AccessToken, oidcClient.Spec.Realm, *clientRepresentation)
		if kcErr != nil {
			return
		}
	}

	cred, kcErr := keycloakClient.GetClientSecret(ctx, token.AccessToken, oidcClient.Spec.Realm, *clientRepresentation.ID)
	if err != nil {
		return
	}

	secret = &corev1.Secret{}
	secret.Name = fmt.Sprintf("%s-%s-oidc", oidcClient.Name, oidcClient.Spec.Realm)
	secret.Namespace = oidcClient.Namespace

	_, err = controllerutil.CreateOrUpdate(ctx, r.Client, secret, func() error {
		data := map[string][]byte{
			"client-id":     []byte(oidcClient.Spec.ClientID),
			"client-secret": []byte(*cred.Value),
		}

		secret.Data = data

		return controllerutil.SetOwnerReference(&oidcClient, secret, r.Scheme)
	})
	if err != nil {
		return
	}

	clientScopes, kcErr := keycloakClient.GetClientScopes(ctx, token.AccessToken, oidcClient.Spec.Realm)
	if kcErr != nil {
		return
	}

	var clientScopeID string
	for _, clientScope := range clientScopes {
		if *clientScope.Name == "groups" {
			clientScopeID = *clientScope.ID
		}
	}

	kcErr = keycloakClient.AddDefaultScopeToClient(ctx, token.AccessToken, oidcClient.Spec.Realm, *clientRepresentation.ID, clientScopeID)
	if kcErr != nil {
		return
	}

	conditions.Delete(&oidcClient, meta.ReconcilingCondition)
	conditions.Delete(&oidcClient, meta.StalledCondition)

	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *OIDCClientReconciler) SetupWithManager(mgr ctrl.Manager) error {
	if err := mgr.GetFieldIndexer().IndexField(context.Background(), &corev1alpha1.OIDCClient{}, ".metadata.name", func(rawObj client.Object) []string {
		// Extract the secret name of the Realm system user, if one is provided
		oidcClient := rawObj.(*corev1alpha1.OIDCClient)
		if oidcClient.Spec.Realm == "" {
			return nil
		}
		return []string{fmt.Sprintf("%s-realm", oidcClient.Spec.Realm)}
	}); err != nil {
		return err
	}

	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1alpha1.OIDCClient{}).
		Watches(
			&source.Kind{Type: &corev1.Secret{}},
			handler.EnqueueRequestsFromMapFunc(r.findObjectsForSecret),
			builder.WithPredicates(predicate.ResourceVersionChangedPredicate{}),
		).
		Complete(r)
}

func (r *OIDCClientReconciler) findObjectsForSecret(secret client.Object) []reconcile.Request {
	list := &corev1alpha1.OIDCClientList{}
	listOps := &client.ListOptions{
		Namespace:     secret.GetNamespace(),
		FieldSelector: fields.OneTermEqualSelector(".metadata.name", secret.GetName()),
	}

	err := r.List(context.TODO(), list, listOps)
	if err != nil {
		return []reconcile.Request{}
	}

	requests := make([]reconcile.Request, len(list.Items))
	for i, item := range list.Items {
		requests[i] = reconcile.Request{
			NamespacedName: types.NamespacedName{
				Name:      item.GetName(),
				Namespace: item.GetNamespace(),
			},
		}
	}
	return requests
}
