package core

import (
	"context"
	"fmt"
	"net/url"
	"strings"

	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	pgv1 "github.com/zalando/postgres-operator/pkg/apis/acid.zalan.do/v1"
	corev1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"
)

// PostgresReconciler reconciles a Postgres object
type PostgresReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=core.libre.sh,resources=postgres,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=core.libre.sh,resources=postgres/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=core.libre.sh,resources=postgres/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Postgres object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *PostgresReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	var postgres corev1alpha1.Postgres
	err = r.Client.Get(ctx, req.NamespacedName, &postgres)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if postgres.Spec.Suspend {
		return ctrl.Result{}, nil
	}

	log.Info("Reconciling")

	defer internal.UpdateStatus(ctx, r.Client, &postgres, &err)

	if postgres.Generation != conditions.GetObservedGeneration(&postgres, meta.ReadyCondition) {
		conditions.MarkReconciling(&postgres, meta.ProgressingReason, "Reconciliation in progress")
		err = r.Client.Status().Update(ctx, &postgres)
		if err != nil {
			return
		}
	}

	var bucket corev1alpha1.Bucket
	bucket.SetName("backups")
	bucket.SetNamespace(postgres.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &bucket, func() error {
		bucket.Spec.Suspend = postgres.Spec.Suspend
		bucket.Spec.Provider = corev1alpha1.BucketBackupProvider
		bucket.Spec.Type = corev1alpha1.BucketPrivateType
		return nil
	})
	if err != nil {
		return
	}
	internal.SetBucketNotReady(&postgres, &bucket)
	if !conditions.IsReady(&bucket) {
		return
	}

	// var postgresConfig corev1.Secret
	// postgresConfig.SetName("postgres-pod-config")
	// postgresConfig.SetNamespace(postgres.Namespace)
	// _, err = ctrl.CreateOrUpdate(ctx, r.Client, &postgresConfig, func() error {
	// 	postgresConfig.StringData = map[string]string{
	// 		"AWS_S3_FORCE_PATH_STYLE": "true",
	// 		"WALG_DISABLE_S3_SSE":     "true",
	// 		"USE_WALG_BACKUP":         "true",
	// 		"USE_WALG_RESTORE":        "true",
	// 	}
	// 	return nil
	// })
	// if err != nil {
	// 	return
	// }

	pg := pgv1.Postgresql{
		ObjectMeta: metav1.ObjectMeta{
			Name:      postgres.GetZalandoName(),
			Namespace: postgres.GetNamespace(),
		},
	}
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &pg, func() error {
		pg.Spec.Users = make(map[string]pgv1.UserFlags)
		pg.Spec.Users[postgres.UserName()] = pgv1.UserFlags{
			"superuser",
			"createdb",
		}
		pg.Spec.Databases = make(map[string]string)
		pg.Spec.Databases[postgres.DatabaseName()] = postgres.UserName()
		pg.Spec.TeamID = corev1alpha1.ZalandoTeam
		pg.Spec.NumberOfInstances = 2
		pg.Spec.PostgresqlParam = pgv1.PostgresqlParam{
			PgVersion: "14",
		}
		pg.Spec.Volume = pgv1.Volume{
			Size: "5G",
		}
		// pg.Spec.EnableLogicalBackup = true
		// pg.Spec.LogicalBackupSchedule = "*/5 * * * *"
		pg.Spec.Env = []corev1.EnvVar{
			{
				Name:  "AWS_S3_FORCE_PATH_STYLE",
				Value: "true",
			},
			{
				Name:  "WALG_DISABLE_S3_SSE",
				Value: "true",
			},
			{
				Name:  "USE_WALG_BACKUP",
				Value: "true",
			},
			{
				Name:  "USE_WALG_RESTORE",
				Value: "true",
			},
			{
				Name:  "BACKUP_SCHEDULE",
				Value: "*/5 * * * *",
			},
			{
				Name:  "BACKUP_SCHEDULE",
				Value: "15",
			},
			{
				Name: "WAL_S3_BUCKET",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "bucket",
					},
				},
			},
			{
				Name: "AWS_ENDPOINT",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "url",
					},
				},
			},
			{
				Name: "AWS_ACCESS_KEY_ID",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "accessKey",
					},
				},
			},
			{
				Name: "AWS_SECRET_ACCESS_KEY",
				ValueFrom: &corev1.EnvVarSource{
					SecretKeyRef: &corev1.SecretKeySelector{
						LocalObjectReference: corev1.LocalObjectReference{
							Name: bucket.GetBucketSecretName(),
						},
						Key: "secretKey",
					},
				},
			},
		}
		return controllerutil.SetControllerReference(&postgres, &pg, r.Scheme)
	})
	if err != nil {
		return
	}

	var secret corev1.Secret
	secret.Name = fmt.Sprintf("%s.postgres.libre.sh", postgres.Name)
	secret.Namespace = bucket.Namespace
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &secret, func() error {
		var zalandoSecret corev1.Secret
		err := r.Get(ctx, types.NamespacedName{
			Name:      fmt.Sprintf("%s.%s.credentials.postgresql.acid.zalan.do", postgres.Name, postgres.GetZalandoName()),
			Namespace: pg.Namespace,
		}, &zalandoSecret)
		if err != nil {
			return err
		}

		username := string(zalandoSecret.Data["username"])
		password := string(zalandoSecret.Data["password"])
		var u url.URL
		u.Host = postgres.GetZalandoName()
		u.Scheme = "postgres"
		u.User = url.UserPassword(username, password)
		u.Path = postgres.DatabaseName()
		u.RawQuery = "sslmode=prefer"
		secret.StringData = map[string]string{
			"username": username,
			"password": password,
			"database": postgres.DatabaseName(),
			"port":     "5432",
			"endpoint": postgres.GetZalandoName(),
			"url":      u.String(),
		}
		return controllerutil.SetOwnerReference(&postgres, &secret, r.Scheme)
	})
	if err != nil {
		return
	}

	switch pg.Status.PostgresClusterStatus {
	case pgv1.ClusterStatusAddFailed, pgv1.ClusterStatusSyncFailed, pgv1.ClusterStatusInvalid, pgv1.ClusterStatusUpdateFailed:
		conditions.MarkStalled(&postgres, meta.FailedReason, "Posgresql status is %s", pg.Status.PostgresClusterStatus)
	case pgv1.ClusterStatusRunning:
		conditions.Delete(&postgres, meta.StalledCondition)
		conditions.Delete(&postgres, meta.ReconcilingCondition)
	case "":
	default:
		conditions.MarkStalled(&postgres, meta.ProgressingReason, "Postgres is %s", strings.ToLower(pg.Status.PostgresClusterStatus))
	}

	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *PostgresReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1alpha1.Postgres{}).
		Owns(&pgv1.Postgresql{}).
		Owns(&corev1alpha1.Bucket{}).
		Watches(
			&source.Kind{Type: &corev1alpha1.Bucket{}},
			handler.EnqueueRequestsFromMapFunc(r.findObjectsForBackupBucket),
			builder.WithPredicates(predicate.ResourceVersionChangedPredicate{}),
		).
		Complete(r)
}

func (r *PostgresReconciler) findObjectsForBackupBucket(bucket client.Object) []reconcile.Request {
	list := &corev1alpha1.PostgresList{}
	listOps := &client.ListOptions{
		Namespace: bucket.GetNamespace(),
	}
	err := r.List(context.TODO(), list, listOps)
	if err != nil {
		return []reconcile.Request{}
	}

	requests := make([]reconcile.Request, len(list.Items))
	for i, item := range list.Items {
		requests[i] = reconcile.Request{
			NamespacedName: types.NamespacedName{
				Name:      item.GetName(),
				Namespace: item.GetNamespace(),
			},
		}
	}
	return requests
}
