package core

import (
	"context"
	"fmt"

	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	appv1 "k8s.io/api/apps/v1"
	corev1 "k8s.io/api/core/v1"
	v1 "k8s.io/api/core/v1"
	metav1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/util/intstr"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/log"
)

// RedisReconciler reconciles a Redis object
type RedisReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=core.libre.sh,resources=redis,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=core.libre.sh,resources=redis/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=core.libre.sh,resources=redis/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Redis object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.2/pkg/reconcile
func (r *RedisReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	var redis corev1alpha1.Redis
	err = r.Client.Get(ctx, req.NamespacedName, &redis)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if redis.Spec.Suspend {
		return ctrl.Result{}, nil
	}

	log.Info("Reconciling")

	defer internal.UpdateStatus(ctx, r.Client, &redis, &err)

	if redis.Generation != conditions.GetObservedGeneration(&redis, meta.ReadyCondition) {
		conditions.MarkReconciling(&redis, meta.ProgressingReason, "Reconciling in progress")
		err = r.Client.Status().Update(ctx, &redis)
		if err != nil {
			return
		}
	}

	labels := map[string]string{
		"app.kubernetes.io/name":       "keydb",
		"app.kubernetes.io/instance":   redis.Name,
		"app.kubernetes.io/managed-by": "libre.sh",
	}

	sts := appv1.StatefulSet{}
	sts.Name = fmt.Sprintf("keydb-%s", redis.Name)
	sts.Namespace = redis.Namespace
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &sts, func() error {
		sts.Labels = labels
		replicas := int32(1)
		sts.Spec.Replicas = &replicas
		sts.Spec.Selector = &metav1.LabelSelector{
			MatchLabels: labels,
		}
		probe := v1.Probe{
			ProbeHandler: v1.ProbeHandler{
				TCPSocket: &v1.TCPSocketAction{
					Port: intstr.FromInt(6379),
				},
			},
		}
		sts.Spec.Template = v1.PodTemplateSpec{
			ObjectMeta: metav1.ObjectMeta{
				Labels: labels,
			},
			Spec: v1.PodSpec{
				Containers: []v1.Container{
					{
						Name:           "keydb",
						Image:          "eqalpha/keydb:alpine_x86_64_v6.3.1",
						Args:           []string{"--active-replica", "yes", "--multi-master", "yes"},
						ReadinessProbe: &probe,
						LivenessProbe:  &probe,
					},
				},
			},
		}
		return nil
	})
	if err != nil {
		return
	}

	service := corev1.Service{}
	service.SetName(redis.Name)
	service.SetNamespace(redis.Namespace)
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &service, func() error {
		service.Labels = labels
		service.Spec = v1.ServiceSpec{
			Ports: []v1.ServicePort{
				{
					Port:       6379,
					TargetPort: intstr.FromInt(6379),
				},
			},
		}
		return controllerutil.SetOwnerReference(&redis, &service, r.Scheme)
	})
	if err != nil {
		return
	}

	var secret corev1.Secret
	secret.Name = fmt.Sprintf("%s.redis.libre.sh", redis.Name)
	secret.Namespace = redis.Namespace
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &secret, func() error {
		secret.StringData = map[string]string{
			"endpoint": redis.Name,
		}
		return controllerutil.SetOwnerReference(&redis, &secret, r.Scheme)
	})
	// if err != nil {
	// 	return
	// }

	conditions.Delete(&redis, meta.ReconcilingCondition)
	conditions.Delete(&redis, meta.StalledCondition)

	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *RedisReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1alpha1.Redis{}).
		Complete(r)
}
