package core

import (
	"context"
	"fmt"
	"time"

	"github.com/Nerzal/gocloak/v11"
	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	corev1 "k8s.io/api/core/v1"
	apierrors "k8s.io/apimachinery/pkg/api/errors"
	"k8s.io/apimachinery/pkg/fields"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"

	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
)

// SAMLClientReconciler reconciles a SAMLClient object
type SAMLClientReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=core.libre.sh,resources=samlclients,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=core.libre.sh,resources=samlclients/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=core.libre.sh,resources=samlclients/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the SAMLClient object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *SAMLClientReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	log.Info("reconciling")

	var samlClient corev1alpha1.SAMLClient
	err = r.Client.Get(ctx, req.NamespacedName, &samlClient)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if samlClient.Spec.Suspend {
		// TODO set suspendended in status ?
		return ctrl.Result{}, nil
	}

	var kcErr error

	defer func() {
		if err != nil {
			return
		}
		if kcErr != nil {
			log.Error(kcErr, "Failed to provision saml client")
			result.RequeueAfter = 10 * time.Minute
			conditions.MarkStalled(&samlClient, meta.FailedReason, kcErr.Error())
		}
		if conditions.IsStalled(&samlClient) {
			conditions.MarkFalse(&samlClient, meta.ReadyCondition, meta.FailedReason, conditions.GetMessage(&samlClient, meta.StalledCondition))
		} else if conditions.IsReconciling(&samlClient) {
			conditions.MarkUnknown(&samlClient, meta.ReadyCondition, meta.ProgressingReason, "Is reconciling")
		} else {
			conditions.MarkTrue(&samlClient, meta.ReadyCondition, meta.SucceededReason, "Is provisionned")
		}
		err = r.Client.Status().Update(ctx, &samlClient)
	}()

	if samlClient.Generation != conditions.GetObservedGeneration(&samlClient, meta.ReadyCondition) {
		conditions.MarkReconciling(&samlClient, meta.ProgressingReason, "Reconciliation in progress")
		err = r.Client.Status().Update(ctx, &samlClient)
		if err != nil {
			return
		}
	}

	secret := &corev1.Secret{}

	// TODO
	err = r.Client.Get(ctx, types.NamespacedName{Namespace: samlClient.Namespace, Name: fmt.Sprintf("%s-realm", samlClient.Namespace)}, secret)
	if err != nil {
		if apierrors.IsNotFound(err) {
			conditions.MarkStalled(&samlClient, meta.FailedReason, err.Error() /* "Dependency is not ready" */)
			err = nil
		}
		return
	}

	config, err := internal.GetConfig(ctx, r.Client, samlClient.Namespace)
	if err != nil {
		return
	}

	provider, err := config.GetKeycloak(ctx, r.Client)
	if err != nil {
		return
	}

	samlClient.SetDefaults()

	// TODO this only works for new keycloak version that do not use the auth subpath
	keycloakClient := gocloak.NewClient(provider.Endpoint, gocloak.SetAuthAdminRealms("admin/realms"), gocloak.SetAuthRealms("realms"))
	token, kcErr := keycloakClient.LoginAdmin(ctx, string(secret.Data["username"]), string(secret.Data["password"]), samlClient.Namespace)
	if kcErr != nil {
		return
	}

	getClientsParams := gocloak.GetClientsParams{
		ClientID: &samlClient.Spec.ClientID,
	}

	var samlClientExists bool
	clientRepresentation := &gocloak.Client{}

	clientRepresentations, kcErr := keycloakClient.GetClients(ctx, token.AccessToken, samlClient.Namespace, getClientsParams)
	if kcErr != nil {
		return
	}

	// TODO manage it if returns multiple clients ?
	if len(clientRepresentations) == 1 {
		samlClientExists = true
		clientRepresentation = clientRepresentations[0]
	}

	err = internal.MutateSAMLClientRepresentation(ctx, r.Client, samlClient, clientRepresentation)
	if err != nil {
		if apierrors.IsNotFound(err) {
			conditions.MarkStalled(&samlClient, meta.FailedReason, err.Error() /* "Dependency is not ready" */)
			err = nil
		}
		return
	}
	if !samlClientExists {
		log.Info("creating saml client")
		clientID, kcErr := keycloakClient.CreateClient(ctx, token.AccessToken, samlClient.Namespace, *clientRepresentation)
		if kcErr != nil {
			return
		}
		clientRepresentation.ID = &clientID
	} else {
		log.Info("updating saml client")
		kcErr = keycloakClient.UpdateClient(ctx, token.AccessToken, samlClient.Namespace, *clientRepresentation)
		if kcErr != nil {
			return
		}
	}

	err = r.CreateOrUpdateSecret(ctx, keycloakClient, token.AccessToken, samlClient, provider)
	if err != nil {
		return
	}

	conditions.Delete(&samlClient, meta.ReconcilingCondition)
	conditions.Delete(&samlClient, meta.StalledCondition)

	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *SAMLClientReconciler) SetupWithManager(mgr ctrl.Manager) error {
	if err := mgr.GetFieldIndexer().IndexField(context.Background(), &corev1alpha1.SAMLClient{}, ".metadata.name", func(rawObj client.Object) []string {
		// Extract the secret name of the Realm system user, if one is provided
		samlClient := rawObj.(*corev1alpha1.SAMLClient)
		if samlClient.Namespace == "" {
			return nil
		}
		return []string{fmt.Sprintf("%s-realm", samlClient.Namespace)}
	}); err != nil {
		return err
	}

	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1alpha1.SAMLClient{}).
		Watches(
			&source.Kind{Type: &corev1.Secret{}},
			handler.EnqueueRequestsFromMapFunc(r.findObjectsForSecret),
			builder.WithPredicates(predicate.ResourceVersionChangedPredicate{}),
		).
		Complete(r)
}

func (r *SAMLClientReconciler) findObjectsForSecret(secret client.Object) []reconcile.Request {
	list := &corev1alpha1.SAMLClientList{}
	listOps := &client.ListOptions{
		Namespace:     secret.GetNamespace(),
		FieldSelector: fields.OneTermEqualSelector(".metadata.name", secret.GetName()),
	}

	err := r.List(context.TODO(), list, listOps)
	if err != nil {
		return []reconcile.Request{}
	}

	requests := make([]reconcile.Request, len(list.Items))
	for i, item := range list.Items {
		requests[i] = reconcile.Request{
			NamespacedName: types.NamespacedName{
				Name:      item.GetName(),
				Namespace: item.GetNamespace(),
			},
		}
	}
	return requests
}

func (r *SAMLClientReconciler) CreateOrUpdateSecret(ctx context.Context, c gocloak.GoCloak, token string, samlClient corev1alpha1.SAMLClient, provider internal.Provider) error {
	keyStore, err := c.GetKeyStoreConfig(ctx, token, samlClient.Namespace)
	if err != nil {
		return err
	}
	var cert *string

	for _, k := range *keyStore.Key {
		if *k.Algorithm == "RS256" {
			cert = k.Certificate
		}
	}

	var secret corev1.Secret
	secret.SetName(samlClient.SecretName())
	secret.SetNamespace(samlClient.Namespace)
	_, err = controllerutil.CreateOrUpdate(context.TODO(), r.Client, &secret, func() error {
		data := map[string][]byte{
			"tls.crt":  []byte(*cert),
			"endpoint": []byte(provider.Endpoint),
		}

		secret.Data = data

		return controllerutil.SetOwnerReference(&samlClient, &secret, r.Scheme)
	})

	return err
}
