package internal

import (
	"context"

	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

func SetDependencyNotReady(to conditions.Setter, t string, from conditions.Getter, message string, messageArgs ...interface{}) {
	if !conditions.IsReady(from) {
		reason := conditions.GetReason(from, meta.ReadyCondition)
		if reason == "" {
			reason = meta.ProgressingReason
		}
		conditions.MarkTrue(to, t, reason, message, messageArgs...)
	} else {
		conditions.Delete(to, t)
	}
}

func SetBucketNotReady(to conditions.Setter, from conditions.Getter) {
	SetDependencyNotReady(to, corev1alpha1.BucketNotReadyCondition, from, "Bucket %s is not ready", from.GetName())
}

func SetPostgresNotReady(to conditions.Setter, from conditions.Getter) {
	SetDependencyNotReady(to, corev1alpha1.PostgrestNotReadyCondition, from, "Postgres %s is not ready", from.GetName())
}

func SetSAMLNotReady(to conditions.Setter, from conditions.Getter) {
	SetDependencyNotReady(to, corev1alpha1.SAMLNotReadyCondition, from, "SAML %s is not ready", from.GetName())
}

func SetRedisNotReady(to conditions.Setter, from conditions.Getter) {
	SetDependencyNotReady(to, corev1alpha1.RedisNotReadyCondition, from, "Redis %s is not ready", from.GetName())
}

func UpdateStatus(ctx context.Context, client client.Client, obj conditions.Setter, err *error) {
	if *err != nil {
		return
	}
	conditions.MarkTrue(obj, meta.ReadyCondition, meta.SucceededReason, "Up and running")
	conditions.SetSummary(obj, meta.ReadyCondition,
		conditions.WithNegativePolarityConditions(
			meta.ReconcilingCondition,
			meta.StalledCondition,
			corev1alpha1.PostgrestNotReadyCondition,
			corev1alpha1.BucketNotReadyCondition,
			corev1alpha1.SAMLNotReadyCondition,
		),
	)
	*err = client.Status().Update(ctx, obj)
}
