package internal

import (
	"context"

	"github.com/minio/madmin-go"
	"github.com/minio/minio-go/v7"
	"github.com/minio/minio-go/v7/pkg/credentials"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

type MinioClient struct {
	*minio.Client
	Admin    *madmin.AdminClient
	Provider Provider
}

func GetMinioClient(ctx context.Context, c client.Client, bucket *corev1alpha1.Bucket) (MinioClient, error) {
	config, err := GetConfig(ctx, c, bucket.Namespace)
	if err != nil {
		return MinioClient{}, err
	}
	var provider Provider
	switch bucket.Spec.Provider {
	case corev1alpha1.BucketDataProvider:
		provider, err = config.GetBackupObjectStore(ctx, c)
		if err != nil {
			return MinioClient{}, err
		}
	case corev1alpha1.BucketBackupProvider:
		provider, err = config.GetDataObjectStore(ctx, c)
		if err != nil {
			return MinioClient{}, err
		}
	}

	minioClient, err := minio.New(provider.Endpoint, &minio.Options{
		Creds:  credentials.NewStaticV4(provider.Username, provider.Password, ""),
		Secure: !provider.Insecure,
	})
	if err != nil {
		return MinioClient{}, err
	}
	minioAdminClient, err := madmin.New(provider.Endpoint, provider.Username, provider.Password, !provider.Insecure)
	if err != nil {
		return MinioClient{}, err
	}
	return MinioClient{
		Client:   minioClient,
		Admin:    minioAdminClient,
		Provider: provider,
	}, nil
}
