package core

import (
	"context"
	"fmt"

	corev1 "k8s.io/api/core/v1"
	v1 "k8s.io/apimachinery/pkg/apis/meta/v1"
	"k8s.io/apimachinery/pkg/apis/meta/v1/unstructured"
	"k8s.io/apimachinery/pkg/types"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

var PostgresTeam = "pg"
var PostgresSecretSuffix = "credentials.postgresql.acid.zalan.do"

type Postgres struct {
	Name      string
	Namespace string
}

type PostgresCredential struct {
	Host     string
	Database string
	User     string
	Password string
}

func (pg *Postgres) GetResourceName() string {
	return fmt.Sprintf("%s-%s", PostgresTeam, pg.Name)
}

func (pg *Postgres) Reconcile(ctx context.Context, client client.Client, references []v1.OwnerReference) (bool, error) {
	postgres := unstructured.Unstructured{}
	postgres.SetAPIVersion("acid.zalan.do/v1")
	postgres.SetKind("postgresql")
	postgres.SetName(pg.GetResourceName())
	postgres.SetNamespace(pg.Namespace)
	_, err := ctrl.CreateOrUpdate(ctx, client, &postgres, func() error {
		users := make(map[string][]string)
		databases := make(map[string]string)
		users[pg.Name] = []string{
			"superuser",
			"createdb",
		}
		databases[pg.Name] = pg.Name
		postgres.Object["spec"] = map[string]interface{}{
			"teamId":            PostgresTeam,
			"numberOfInstances": 2,
			"postgresql": map[string]interface{}{
				"version": "13",
			},
			"volume": map[string]interface{}{
				"size": "5G",
			},
			"databases": databases,
			"users":     users,
		}
		postgres.SetOwnerReferences(references)
		return nil
	})
	if err != nil {
		return false, err
	}

	return !(postgres.Object["status"] == nil || postgres.Object["status"].(map[string]interface{})["PostgresClusterStatus"] != "Running"), nil
}

func (pg *Postgres) GetCredentials(ctx context.Context, client client.Client) (PostgresCredential, error) {
	secret := corev1.Secret{}
	err := client.Get(ctx, types.NamespacedName{
		Namespace: pg.Namespace,
		Name:      fmt.Sprintf("%s.%s.%s", pg.Name, pg.GetResourceName(), PostgresSecretSuffix),
	}, &secret)
	if err != nil {
		return PostgresCredential{}, err
	}

	return PostgresCredential{
		Host:     pg.GetResourceName(),
		Database: pg.Name,
		User:     string(secret.Data["username"]),
		Password: string(secret.Data["password"]),
	}, nil
}

func (credential PostgresCredential) URL() string {
	return fmt.Sprintf("postgres://%s:%s@%s/%s?sslmode=require", credential.User, credential.Password, credential.Host, credential.Database)
}
