package core

import (
	"bytes"
	"context"
	"net/url"
	"text/template"
	"time"

	"github.com/fluxcd/pkg/apis/meta"
	"github.com/fluxcd/pkg/runtime/conditions"
	"github.com/minio/madmin-go"
	"github.com/minio/minio-go/v7"
	"github.com/minio/minio-go/v7/pkg/credentials"
	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/types"
	"k8s.io/apimachinery/pkg/util/rand"
	appsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	corev1alpha1 "libre.sh/controller/apis/core/v1alpha1"
	"libre.sh/controller/internal"
	ctrl "sigs.k8s.io/controller-runtime"
	"sigs.k8s.io/controller-runtime/pkg/builder"
	"sigs.k8s.io/controller-runtime/pkg/client"
	"sigs.k8s.io/controller-runtime/pkg/controller/controllerutil"
	"sigs.k8s.io/controller-runtime/pkg/handler"
	"sigs.k8s.io/controller-runtime/pkg/log"
	"sigs.k8s.io/controller-runtime/pkg/predicate"
	"sigs.k8s.io/controller-runtime/pkg/reconcile"
	"sigs.k8s.io/controller-runtime/pkg/source"
)

const ownerPolicy = `
{
	"Version": "2012-10-17",
	"Statement": [
	  {
		"Effect": "Allow",
		"Action": ["s3:*"],
		"Resource": [
		  "arn:aws:s3:::${aws:username}/*",
		  "arn:aws:s3:::${aws:username}"
		]
	  }
	]
}
`

const cdnPolicy = `
{
	"Version": "2012-10-17",
	"Statement": [
	  {
		"Action": ["s3:GetBucketLocation"],
		"Effect": "Allow",
		"Principal": { "AWS": ["*"] },
		"Resource": ["arn:aws:s3:::{{ . }}"]
	  },
	  {
		"Action": ["s3:GetObject"],
		"Effect": "Allow",
		"Principal": { "AWS": ["*"] },
		"Resource": ["arn:aws:s3:::{{ . }}/*","arn:aws:s3:::{{ . }}/**"]
	  }
	]
  }  
`

// BucketReconciler reconciles a Bucket object
type BucketReconciler struct {
	client.Client
	Scheme *runtime.Scheme
}

//+kubebuilder:rbac:groups=internal.libre.sh,resources=buckets,verbs=get;list;watch;create;update;patch;delete
//+kubebuilder:rbac:groups=internal.libre.sh,resources=buckets/status,verbs=get;update;patch
//+kubebuilder:rbac:groups=internal.libre.sh,resources=buckets/finalizers,verbs=update

// Reconcile is part of the main kubernetes reconciliation loop which aims to
// move the current state of the cluster closer to the desired state.
// TODO(user): Modify the Reconcile function to compare the state specified by
// the Bucket object against the actual cluster state, and then
// perform operations to make the cluster state reflect the state specified by
// the user.
//
// For more details, check Reconcile and its Result here:
// - https://pkg.go.dev/sigs.k8s.io/controller-runtime@v0.12.1/pkg/reconcile
func (r *BucketReconciler) Reconcile(ctx context.Context, req ctrl.Request) (result ctrl.Result, err error) {
	log := log.FromContext(ctx)

	var bucket corev1alpha1.Bucket
	err = r.Client.Get(ctx, req.NamespacedName, &bucket)
	if err != nil {
		return ctrl.Result{}, client.IgnoreNotFound(err)
	}

	if bucket.Spec.Suspend {
		return ctrl.Result{}, nil
	}

	log.Info("Reconciling")

	var minioErr error

	defer func() {
		if err == nil && minio.IsNetworkOrHostDown(minioErr, false) {
			err = minioErr
		}
		if err != nil {
			return
		}
		if minioErr != nil {
			log.Error(minioErr, "Failed to provision bucket")
			result.RequeueAfter = 10 * time.Minute
			conditions.MarkStalled(&bucket, meta.FailedReason, minioErr.Error())
		} else {
			conditions.Delete(&bucket, meta.StalledCondition)
		}
		if conditions.IsStalled(&bucket) {
			conditions.MarkFalse(&bucket, meta.ReadyCondition, meta.FailedReason, "Is stalled")
		} else if conditions.IsReconciling(&bucket) {
			conditions.MarkUnknown(&bucket, meta.ReadyCondition, meta.ProgressingReason, "Is reconciling")
		} else {
			conditions.MarkTrue(&bucket, meta.ReadyCondition, meta.SucceededReason, "Is provisionned")
		}
		err = r.Client.Status().Update(ctx, &bucket)
	}()

	if bucket.Generation != conditions.GetObservedGeneration(&bucket, meta.ReadyCondition) {
		conditions.MarkReconciling(&bucket, meta.ProgressingReason, "Reconciling in progress")
		err = r.Client.Status().Update(ctx, &bucket)
		if err != nil {
			return
		}
	}

	config, minioErr := internal.GetConfig(ctx, r.Client, bucket.Namespace)
	if minioErr != nil {
		return
	}
	var provider internal.Provider
	switch bucket.Spec.Provider {
	case corev1alpha1.BucketDataProvider:
		provider, minioErr = config.GetBackupObjectStore(ctx, r.Client)
		if minioErr != nil {
			return
		}
	case corev1alpha1.BucketBackupProvider:
		provider, minioErr = config.GetDataObjectStore(ctx, r.Client)
		if minioErr != nil {
			return
		}
	}

	minioClient, minioErr := minio.New(provider.Endpoint, &minio.Options{
		Creds:  credentials.NewStaticV4(provider.Username, provider.Password, ""),
		Secure: !provider.Insecure,
	})
	if minioErr != nil {
		return
	}
	minioAdminClient, minioErr := madmin.New(provider.Endpoint, provider.Username, provider.Password, !provider.Insecure)
	if minioErr != nil {
		return
	}

	buckerExits, minioErr := minioClient.BucketExists(ctx, bucket.GetBucketName())
	if minioErr != nil {
		return
	}
	if !buckerExits {
		// TODO region
		minioErr = minioClient.MakeBucket(ctx, bucket.GetBucketName(), minio.MakeBucketOptions{})
		if minioErr != nil {
			return
		}

	}

	var secret corev1.Secret
	secret.Name = bucket.GetBucketSecretName()
	secret.Namespace = bucket.Namespace
	_, err = ctrl.CreateOrUpdate(ctx, r.Client, &secret, func() error {
		secretKey := string(secret.Data["secretKey"])
		if secretKey == "" {
			secretKey = rand.String(32)
		}
		var url url.URL
		url.Host = provider.Endpoint
		if provider.Insecure {
			url.Scheme = "http"
		} else {
			url.Scheme = "https"
		}
		secret.StringData = map[string]string{
			"accessKey": bucket.GetBucketName(),
			"secretKey": secretKey,
			"endpoint":  provider.Endpoint,
			"url":       url.String(),
			"bucket":    bucket.GetBucketName(),
		}
		return controllerutil.SetOwnerReference(&bucket, &secret, r.Scheme)
	})
	if err != nil {
		return
	}

	minioErr = minioAdminClient.AddUser(ctx, bucket.GetBucketName(), string(secret.Data["secretKey"]))
	if minioErr != nil {
		return
	}

	minioErr = minioAdminClient.AddCannedPolicy(ctx, "owner", []byte(ownerPolicy))
	if minioErr != nil {
		return
	}

	minioErr = minioAdminClient.SetPolicy(ctx, "owner", bucket.GetBucketName(), false)
	if minioErr != nil {
		return
	}

	switch bucket.Spec.Type {
	case corev1alpha1.BucketCDNType:
		var tmpl *template.Template
		tmpl, minioErr = template.New("policy").Parse(cdnPolicy)
		if minioErr != nil {
			return
		}
		buff := bytes.Buffer{}
		minioErr = tmpl.Execute(&buff, bucket.GetBucketName())
		if minioErr != nil {
			return
		}
		minioErr = minioClient.SetBucketPolicy(ctx, bucket.GetBucketName(), buff.String())
		if minioErr != nil {
			return
		}
	case corev1alpha1.BucketPrivateType:
		minioErr = minioClient.SetBucketPolicy(ctx, bucket.GetBucketName(), "")
		if minioErr != nil {
			return
		}
	}

	// minioErr = fmt.Errorf("mean error")
	conditions.Delete(&bucket, meta.ReconcilingCondition)
	return
}

// SetupWithManager sets up the controller with the Manager.
func (r *BucketReconciler) SetupWithManager(mgr ctrl.Manager) error {
	return ctrl.NewControllerManagedBy(mgr).
		For(&corev1alpha1.Bucket{}).
		Watches(
			&source.Kind{Type: &appsv1alpha1.LibreConfig{}},
			handler.EnqueueRequestsFromMapFunc(r.findObjectsForLibreConfig),
			builder.WithPredicates(predicate.ResourceVersionChangedPredicate{}),
		).
		Complete(r)
}

func (r *BucketReconciler) findObjectsForLibreConfig(libreConf client.Object) []reconcile.Request {
	list := &corev1alpha1.BucketList{}
	listOps := &client.ListOptions{}
	if libreConf.GetNamespace() != internal.GlobalNS {
		listOps.Namespace = libreConf.GetNamespace()
	}
	err := r.List(context.TODO(), list, listOps)
	if err != nil {
		return []reconcile.Request{}
	}

	requests := make([]reconcile.Request, len(list.Items))
	for i, item := range list.Items {
		requests[i] = reconcile.Request{
			NamespacedName: types.NamespacedName{
				Name:      item.GetName(),
				Namespace: item.GetNamespace(),
			},
		}
	}
	return requests
}
