package internal

import (
	"context"
	"errors"

	corev1 "k8s.io/api/core/v1"
	"k8s.io/apimachinery/pkg/types"
	libreappsv1alpha1 "libre.sh/controller/apis/apps/v1alpha1"
	"sigs.k8s.io/controller-runtime/pkg/client"
)

const GlobalNS = "libresh-system"

type Provider struct {
	Endpoint string
	Insecure bool
	Username string
	Password string
}

type Config struct {
	namespace string
	global    *libreappsv1alpha1.LibreConfigSpec
	current   *libreappsv1alpha1.LibreConfigSpec
}

func GetConfig(ctx context.Context, c client.Client, ns string) (Config, error) {
	// TODO use constant
	global, err := getNsConfig(ctx, c, GlobalNS)
	if err != nil {
		return Config{}, err
	}

	current, err := getNsConfig(ctx, c, ns)
	if err != nil {
		return Config{}, err
	}

	if global == nil && current == nil {
		return Config{}, errors.New("no libre config provided")
	}

	return Config{
		namespace: ns,
		global:    global,
		current:   current,
	}, nil
}

func getNsConfig(ctx context.Context, c client.Client, ns string) (*libreappsv1alpha1.LibreConfigSpec, error) {
	var configs libreappsv1alpha1.LibreConfigList
	err := c.List(ctx, &configs, client.InNamespace(ns))
	if err != nil {
		return nil, err
	}

	var config *libreappsv1alpha1.LibreConfigSpec
	if len(configs.Items) > 0 {
		config = &configs.Items[0].Spec
	}

	return config, nil
}

func (config *Config) GetDataObjectStore(ctx context.Context, c client.Client) (Provider, error) {
	if config.current != nil && config.current.DataObjectStore != nil {
		return toProvider(ctx, c, config.namespace, *config.current.DataObjectStore)
	}
	if config.global != nil && config.global.DataObjectStore != nil {
		return toProvider(ctx, c, GlobalNS, *config.global.DataObjectStore)
	}
	return Provider{}, errors.New("no object store configured")
}

func (config *Config) GetBackupObjectStore(ctx context.Context, c client.Client) (Provider, error) {
	if config.current != nil && config.current.BackupObjectStore != nil {
		return toProvider(ctx, c, config.namespace, *config.current.BackupObjectStore)
	}
	if config.global != nil && config.global.BackupObjectStore != nil {
		return toProvider(ctx, c, GlobalNS, *config.global.BackupObjectStore)
	}
	return config.GetDataObjectStore(ctx, c)
}

func toProvider(ctx context.Context, c client.Client, ns string, p libreappsv1alpha1.Provider) (Provider, error) {
	var provider Provider
	provider.Endpoint = p.Endpoint
	provider.Insecure = p.Insecure
	var secret corev1.Secret
	err := c.Get(ctx, types.NamespacedName{Namespace: ns, Name: p.Credentials.Name}, &secret)
	if err != nil {
		return provider, err
	}
	provider.Username = string(secret.Data["username"])
	provider.Password = string(secret.Data["password"])
	return provider, nil
}
