package sh.libre.scim.core;

import com.google.common.net.HttpHeaders;
import de.captaingoldfish.scim.sdk.client.ScimClientConfig;
import de.captaingoldfish.scim.sdk.client.ScimRequestBuilder;
import de.captaingoldfish.scim.sdk.client.http.BasicAuth;
import de.captaingoldfish.scim.sdk.client.response.ServerResponse;
import de.captaingoldfish.scim.sdk.common.exceptions.ResponseException;
import de.captaingoldfish.scim.sdk.common.resources.ResourceNode;
import de.captaingoldfish.scim.sdk.common.response.ListResponse;
import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.github.resilience4j.retry.RetryRegistry;
import jakarta.persistence.EntityManager;
import jakarta.persistence.NoResultException;
import jakarta.ws.rs.ProcessingException;
import org.jboss.logging.Logger;
import org.keycloak.component.ComponentModel;
import org.keycloak.connections.jpa.JpaConnectionProvider;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RoleMapperModel;
import org.keycloak.storage.user.SynchronizationResult;
import sh.libre.scim.jpa.ScimResource;

import java.util.HashMap;
import java.util.Map;


public class ScimClient {
    protected final Logger LOGGER = Logger.getLogger(ScimClient.class);
    protected final ScimRequestBuilder scimRequestBuilder;
    protected final RetryRegistry registry;
    protected final KeycloakSession session;
    protected final String contentType;
    protected final ComponentModel model;
    protected final String scimApplicationBaseUrl;
    protected final Map<String, String> defaultHeaders;
    protected final Map<String, String> expectedResponseHeaders;

    public ScimClient(ComponentModel model, KeycloakSession session) {
        this.model = model;
        this.contentType = model.get("content-type");
        this.session = session;
        this.scimApplicationBaseUrl = model.get("endpoint");
        this.defaultHeaders = new HashMap<>();
        this.expectedResponseHeaders = new HashMap<>();

        switch (model.get("auth-mode")) {
            case "BEARER":
                defaultHeaders.put(HttpHeaders.AUTHORIZATION,
                        BearerAuthentication());
                break;
            case "BASIC_AUTH":
                defaultHeaders.put(HttpHeaders.AUTHORIZATION,
                        BasicAuthentication());
                break;
        }

        defaultHeaders.put(HttpHeaders.CONTENT_TYPE, contentType);

        scimRequestBuilder = new ScimRequestBuilder(scimApplicationBaseUrl, genScimClientConfig());

        RetryConfig retryConfig = RetryConfig.custom()
                .maxAttempts(10)
                .intervalFunction(IntervalFunction.ofExponentialBackoff())
                .retryExceptions(ProcessingException.class)
                .build();

        registry = RetryRegistry.of(retryConfig);
    }

    protected String BasicAuthentication() {
        return BasicAuth.builder()
                .username(model.get("auth-user"))
                .password(model.get("auth-pass"))
                .build()
                .getAuthorizationHeaderValue();
    }

    protected ScimClientConfig genScimClientConfig() {
        return ScimClientConfig.builder()
                .httpHeaders(defaultHeaders)
                .connectTimeout(5)
                .requestTimeout(5)
                .socketTimeout(5)
                .expectedHttpResponseHeaders(expectedResponseHeaders)
                .hostnameVerifier((s, sslSession) -> true)
                .build();
    }

    protected String BearerAuthentication() {
        return "Bearer " + model.get("auth-pass");
    }

    protected EntityManager getEM() {
        return session.getProvider(JpaConnectionProvider.class).getEntityManager();
    }

    protected String getRealmId() {
        return session.getContext().getRealm().getId();
    }

    protected <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> A getAdapter(
            Class<A> aClass) {
        try {
            return aClass.getDeclaredConstructor(KeycloakSession.class, String.class)
                    .newInstance(session, this.model.getId());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void create(Class<A> aClass,
                                                                                                    M kcModel) {
        A adapter = getAdapter(aClass);
        adapter.apply(kcModel);
        if (adapter.skip)
            return;
        // If mapping exist then it was created by import so skip.
        if (adapter.query("findById", adapter.getId()).getResultList().size() != 0) {
            return;
        }
        Retry retry = registry.retry("create-" + adapter.getId());

        ServerResponse<S> response = retry.executeSupplier(() -> {
            try {
                return scimRequestBuilder
                        .create(adapter.getResourceClass(), adapter.getSCIMEndpoint())
                        .setResource(adapter.toSCIM())
                        .sendRequest();
            } catch (ResponseException e) {
                throw new RuntimeException(e);
            }
        });

        if (!response.isSuccess()) {
            LOGGER.warn(response.getResponseBody());
            LOGGER.warn(response.getHttpStatus());
        }

        adapter.apply(response.getResource());
        adapter.saveMapping();
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void replace(Class<A> aClass,
                                                                                                     M kcModel) {
        A adapter = getAdapter(aClass);
        try {
            adapter.apply(kcModel);
            if (adapter.skip)
                return;
            ScimResource resource = adapter.query("findById", adapter.getId()).getSingleResult();
            adapter.apply(resource);
            Retry retry = registry.retry("replace-" + adapter.getId());
            ServerResponse<S> response = retry.executeSupplier(() -> {
                try {
                    return scimRequestBuilder
                            .update(adapter.getResourceClass(), adapter.getSCIMEndpoint(), adapter.getExternalId())
                            .setResource(adapter.toSCIM())
                            .sendRequest();
                } catch (ResponseException e) {
                    throw new RuntimeException(e);
                }
            });
            if (!response.isSuccess()) {
                LOGGER.warn(response.getResponseBody());
                LOGGER.warn(response.getHttpStatus());
            }
        } catch (NoResultException e) {
            LOGGER.warnf("failed to replace resource %s, scim mapping not found", adapter.getId());
        } catch (Exception e) {
            LOGGER.error(e);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void delete(Class<A> aClass,
                                                                                                    String id) {
        A adapter = getAdapter(aClass);
        adapter.setId(id);

        try {
            ScimResource resource = adapter.query("findById", adapter.getId()).getSingleResult();
            adapter.apply(resource);

            Retry retry = registry.retry("delete-" + id);

            ServerResponse<S> response = retry.executeSupplier(() -> {
                try {
                    return scimRequestBuilder.delete(adapter.getResourceClass(), adapter.getSCIMEndpoint(), adapter.getExternalId())
                            .sendRequest();
                } catch (ResponseException e) {
                    throw new RuntimeException(e);
                }
            });

            if (!response.isSuccess()) {
                LOGGER.warn(response.getResponseBody());
                LOGGER.warn(response.getHttpStatus());
            }

            getEM().remove(resource);

        } catch (NoResultException e) {
            LOGGER.warnf("Failed to delete resource %s, scim mapping not found", id);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void refreshResources(
            Class<A> aClass,
            SynchronizationResult syncRes) {
        LOGGER.info("Refresh resources");
        getAdapter(aClass).getResourceStream().forEach(resource -> {
            A adapter = getAdapter(aClass);
            adapter.apply(resource);
            LOGGER.infof("Reconciling local resource %s", adapter.getId());
            if (!adapter.skipRefresh()) {
                ScimResource mapping = adapter.getMapping();
                if (mapping == null) {
                    LOGGER.info("Creating it");
                    this.create(aClass, resource);
                } else {
                    LOGGER.info("Replacing it");
                    this.replace(aClass, resource);
                }
                syncRes.increaseUpdated();
            }
        });

    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void importResources(
            Class<A> aClass, SynchronizationResult syncRes) {
        LOGGER.info("Import");
        try {
            A adapter = getAdapter(aClass);
            ServerResponse<ListResponse<S>> response = scimRequestBuilder.list(adapter.getResourceClass(), adapter.getSCIMEndpoint()).get().sendRequest();
            ListResponse<S> resourceTypeListResponse = response.getResource();

            for (S resource : resourceTypeListResponse.getListedResources()) {
                try {
                    LOGGER.infof("Reconciling remote resource %s", resource);
                    adapter = getAdapter(aClass);
                    adapter.apply(resource);

                    ScimResource mapping = adapter.getMapping();
                    if (mapping != null) {
                        adapter.apply(mapping);
                        if (adapter.entityExists()) {
                            LOGGER.info("Valid mapping found, skipping");
                            continue;
                        } else {
                            LOGGER.info("Delete a dangling mapping");
                            adapter.deleteMapping();
                        }
                    }

                    Boolean mapped = adapter.tryToMap();
                    if (mapped) {
                        LOGGER.info("Matched");
                        adapter.saveMapping();
                    } else {
                        switch (this.model.get("sync-import-action")) {
                            case "CREATE_LOCAL":
                                LOGGER.info("Create local resource");
                                try {
                                    adapter.createEntity();
                                    adapter.saveMapping();
                                    syncRes.increaseAdded();
                                } catch (Exception e) {
                                    LOGGER.error(e);
                                }
                                break;
                            case "DELETE_REMOTE":
                                LOGGER.info("Delete remote resource");
                                scimRequestBuilder
                                        .delete(adapter.getResourceClass(), adapter.getSCIMEndpoint(), resource.getId().get())
                                        .sendRequest();
                                syncRes.increaseRemoved();
                                break;
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error(e);
                    e.printStackTrace();
                    syncRes.increaseFailed();
                }
            }
        } catch (ResponseException e) {
            throw new RuntimeException(e);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void sync(Class<A> aClass,
                                                                                                  SynchronizationResult syncRes) {
        if (this.model.get("sync-import", false)) {
            this.importResources(aClass, syncRes);
        }
        if (this.model.get("sync-refresh", false)) {
            this.refreshResources(aClass, syncRes);
        }
    }

    public void close() {
        scimRequestBuilder.close();
    }
}
