package sh.libre.scim.storage;

import de.captaingoldfish.scim.sdk.common.constants.HttpHeader;
import jakarta.ws.rs.core.MediaType;
import org.apache.commons.lang3.BooleanUtils;
import org.jboss.logging.Logger;
import org.keycloak.component.ComponentModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.KeycloakSessionFactory;
import org.keycloak.models.KeycloakSessionTask;
import org.keycloak.models.utils.KeycloakModelUtils;
import org.keycloak.provider.ProviderConfigProperty;
import org.keycloak.provider.ProviderConfigurationBuilder;
import org.keycloak.storage.UserStorageProviderFactory;
import org.keycloak.storage.UserStorageProviderModel;
import org.keycloak.storage.user.ImportSynchronization;
import org.keycloak.storage.user.SynchronizationResult;
import org.keycloak.timer.TimerProvider;
import sh.libre.scim.core.GroupAdapter;
import sh.libre.scim.core.ScimDispatcher;
import sh.libre.scim.core.UserAdapter;

import java.util.Date;
import java.util.List;

public class ScimStorageProviderFactory
        implements UserStorageProviderFactory<ScimStorageProvider>, ImportSynchronization {

    private final Logger LOGGER = Logger.getLogger(ScimStorageProviderFactory.class);

    public static final String ID = "scim";

    private static final List<ProviderConfigProperty> CONFIG_METADATA;

    static {
        CONFIG_METADATA = ProviderConfigurationBuilder.create()
                .property()
                .name("endpoint")
                .type(ProviderConfigProperty.STRING_TYPE)
                .required(true)
                .label("SCIM 2.0 endpoint")
                .helpText("External SCIM 2.0 base " +
                        "URL (/ServiceProviderConfig  /Schemas and /ResourcesTypes should be accessible)")
                .add()
                .property()
                .name("content-type")
                .type(ProviderConfigProperty.LIST_TYPE)
                .label("Endpoint content type")
                .helpText("Only used when endpoint doesn't support application/scim+json")
                .options(MediaType.APPLICATION_JSON, HttpHeader.SCIM_CONTENT_TYPE)
                .defaultValue(HttpHeader.SCIM_CONTENT_TYPE)
                .add()
                .property()
                .name("auth-mode")
                .type(ProviderConfigProperty.LIST_TYPE)
                .label("Auth mode")
                .helpText("Select the authorization mode")
                .options("NONE", "BASIC_AUTH", "BEARER")
                .defaultValue("NONE")
                .add()
                .property()
                .name("auth-user")
                .type(ProviderConfigProperty.STRING_TYPE)
                .label("Auth username")
                .helpText("Required for basic authentication.")
                .add()
                .property()
                .name("auth-pass")
                .type(ProviderConfigProperty.PASSWORD)
                .label("Auth password/token")
                .helpText("Password or token required for basic or bearer authentication.")
                .add()
                .property()
                .name("propagation-user")
                .type(ProviderConfigProperty.BOOLEAN_TYPE)
                .label("Enable user propagation")
                .helpText("Should operation on users be propagated to this provider?")
                .defaultValue(BooleanUtils.TRUE)
                .add()
                .property()
                .name("propagation-group")
                .type(ProviderConfigProperty.BOOLEAN_TYPE)
                .label("Enable group propagation")
                .helpText("Should operation on groups be propagated to this provider?")
                .defaultValue(BooleanUtils.TRUE)
                .add()
                .property()
                .name("sync-import")
                .type(ProviderConfigProperty.BOOLEAN_TYPE)
                .label("Enable import during sync")
                .add()
                .property()
                .name("sync-import-action")
                .type(ProviderConfigProperty.LIST_TYPE)
                .label("Import action")
                .helpText("What to do when the user doesn't exists in Keycloak.")
                .options("NOTHING", "CREATE_LOCAL", "DELETE_REMOTE")
                .defaultValue("CREATE_LOCAL")
                .add()
                .property()
                .name("sync-refresh")
                .type(ProviderConfigProperty.BOOLEAN_TYPE)
                .label("Enable refresh during sync")
                .add()
                .build();
    }

    @Override
    public ScimStorageProvider create(KeycloakSession session, ComponentModel model) {
        LOGGER.info("create");
        return new ScimStorageProvider();
    }

    @Override
    public String getId() {
        return ID;
    }

    @Override
    public List<ProviderConfigProperty> getConfigProperties() {
        return CONFIG_METADATA;
    }

    @Override
    public SynchronizationResult sync(KeycloakSessionFactory sessionFactory, String realmId,
                                      UserStorageProviderModel model) {
        LOGGER.info("sync");
        var result = new SynchronizationResult();
        KeycloakModelUtils.runJobInTransaction(sessionFactory, new KeycloakSessionTask() {

            @Override
            public void run(KeycloakSession session) {
                var realm = session.realms().getRealm(realmId);
                session.getContext().setRealm(realm);
                var dispatcher = new ScimDispatcher(session);
                if (BooleanUtils.TRUE.equals(model.get("propagation-user"))) {
                    dispatcher.runOne(model, (client) -> client.sync(UserAdapter.class, result));
                }
                if (BooleanUtils.TRUE.equals(model.get("propagation-group"))) {
                    dispatcher.runOne(model, (client) -> client.sync(GroupAdapter.class, result));
                }
            }

        });

        return result;

    }

    @Override
    public SynchronizationResult syncSince(Date lastSync, KeycloakSessionFactory sessionFactory, String realmId,
                                           UserStorageProviderModel model) {
        return this.sync(sessionFactory, realmId, model);
    }

    @Override
    public void postInit(KeycloakSessionFactory factory) {
        var timer = factory.create().getProvider(TimerProvider.class);
        timer.scheduleTask(taskSession -> {
            for (var realm : taskSession.realms().getRealmsStream().toList()) {
                KeycloakModelUtils.runJobInTransaction(factory, new KeycloakSessionTask() {
                    @Override
                    public void run(KeycloakSession session) {
                        session.getContext().setRealm(realm);
                        var dispatcher = new ScimDispatcher(session);
                        for (var group : session.groups().getGroupsStream(realm)
                                .filter(x -> BooleanUtils.TRUE.equals(x.getFirstAttribute("scim-dirty"))).toList()) {
                            LOGGER.debug(group.getName() + " is dirty");
                            dispatcher.run(ScimDispatcher.SCOPE_GROUP,
                                    (client) -> client.replace(GroupAdapter.class, group));
                            group.removeAttribute("scim-dirty");
                        }
                    }

                });
            }
        }, 30000, "scim-background");
    }
}
