package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.common.exceptions.ResponseException;
import de.captaingoldfish.scim.sdk.common.resources.ResourceNode;
import de.captaingoldfish.scim.sdk.common.resources.complex.Meta;
import org.jboss.logging.Logger;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RoleMapperModel;
import org.keycloak.storage.user.SynchronizationResult;
import sh.libre.scim.jpa.ScimResource;
import sh.libre.scim.jpa.ScimResourceDao;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public abstract class AbstractScimService<RMM extends RoleMapperModel, S extends ResourceNode> implements AutoCloseable {

    private static final Logger LOGGER = Logger.getLogger(AbstractScimService.class);

    private final KeycloakSession keycloakSession;

    private final ScrimProviderConfiguration scimProviderConfiguration;

    private final ScimResourceType type;

    private final ScimClient<S> scimClient;

    protected AbstractScimService(KeycloakSession keycloakSession, ScrimProviderConfiguration scimProviderConfiguration, ScimResourceType type) {
        this.keycloakSession = keycloakSession;
        this.scimProviderConfiguration = scimProviderConfiguration;
        this.type = type;
        this.scimClient = ScimClient.open(scimProviderConfiguration, type);
    }

    public void create(RMM roleMapperModel) throws ScimPropagationException {
        boolean skip = isSkip(roleMapperModel);
        if (skip)
            return;
        // If mapping exist then it was created by import so skip.
        KeycloakId id = getId(roleMapperModel);
        if (findById(id).isPresent()) {
            return;
        }
        S scimForCreation = toScimForCreation(roleMapperModel);
        EntityOnRemoteScimId externalId = scimClient.create(id, scimForCreation);
        createMapping(id, externalId);
    }

    protected abstract S toScimForCreation(RMM roleMapperModel);

    protected abstract KeycloakId getId(RMM roleMapperModel);

    protected abstract boolean isSkip(RMM roleMapperModel);

    private void createMapping(KeycloakId keycloakId, EntityOnRemoteScimId externalId) {
        getScimResourceDao().create(keycloakId, externalId, type);
    }

    protected ScimResourceDao getScimResourceDao() {
        return ScimResourceDao.newInstance(getKeycloakSession(), scimProviderConfiguration.getId());
    }

    private Optional<ScimResource> findById(KeycloakId keycloakId) {
        return getScimResourceDao().findById(keycloakId, type);
    }

    private KeycloakSession getKeycloakSession() {
        return keycloakSession;
    }

    public void replace(RMM roleMapperModel) throws ScimPropagationException {
        try {
            if (isSkip(roleMapperModel))
                return;
            KeycloakId id = getId(roleMapperModel);
            Optional<EntityOnRemoteScimId> entityOnRemoteScimId = findById(id)
                    .map(ScimResource::getExternalIdAsEntityOnRemoteScimId);
            entityOnRemoteScimId
                    .ifPresentOrElse(
                            externalId -> doReplace(roleMapperModel, externalId),
                            () -> {
                                // TODO Exception Handling : should we throw a ScimPropagationException here ?
                                LOGGER.warnf("failed to replace resource %s, scim mapping not found", id);
                            }
                    );
        } catch (Exception e) {
            throw new ScimPropagationException("[SCIM] Error while replacing SCIM resource", e);
        }
    }

    private void doReplace(RMM roleMapperModel, EntityOnRemoteScimId externalId) {
        S scimForReplace = toScimForReplace(roleMapperModel, externalId);
        scimClient.replace(externalId, scimForReplace);
    }

    protected abstract S toScimForReplace(RMM roleMapperModel, EntityOnRemoteScimId externalId);

    public void delete(KeycloakId id) throws ScimPropagationException {
        ScimResource resource = findById(id)
                .orElseThrow(() -> new ScimPropagationException("Failed to delete resource %s, scim mapping not found: ".formatted(id)));
        EntityOnRemoteScimId externalId = resource.getExternalIdAsEntityOnRemoteScimId();
        scimClient.delete(externalId);
        getScimResourceDao().delete(resource);
    }

    public void refreshResources(SynchronizationResult syncRes) throws ScimPropagationException {
        LOGGER.info("[SCIM] Refresh resources for endpoint  " + this.getConfiguration().getEndPoint());
        try (Stream<RMM> resourcesStream = getResourceStream()) {
            Set<RMM> resources = resourcesStream.collect(Collectors.toUnmodifiableSet());
            for (RMM resource : resources) {
                KeycloakId id = getId(resource);
                LOGGER.infof("[SCIM] Reconciling local resource %s", id);
                if (isSkipRefresh(resource)) {
                    LOGGER.infof("[SCIM] Skip local resource %s", id);
                    continue;
                }
                if (findById(id).isPresent()) {
                    LOGGER.info("[SCIM] Replacing it");
                    replace(resource);
                } else {
                    LOGGER.info("[SCIM] Creating it");
                    create(resource);
                }
                syncRes.increaseUpdated();
            }
        }
    }

    protected abstract boolean isSkipRefresh(RMM resource);

    protected abstract Stream<RMM> getResourceStream();

    public void importResources(SynchronizationResult syncRes) throws ScimPropagationException {
        LOGGER.info("[SCIM] Import resources for scim endpoint " + this.getConfiguration().getEndPoint());
        try {
            for (S resource : scimClient.listResources()) {
                try {
                    LOGGER.infof("[SCIM] Reconciling remote resource %s", resource);
                    EntityOnRemoteScimId externalId = resource.getId()
                            .map(EntityOnRemoteScimId::new)
                            .orElseThrow(() -> new ScimPropagationException("remote SCIM resource doesn't have an id"));
                    Optional<ScimResource> optionalMapping = getScimResourceDao().findByExternalId(externalId, type);
                    if (optionalMapping.isPresent()) {
                        ScimResource mapping = optionalMapping.get();
                        if (entityExists(mapping.getIdAsKeycloakId())) {
                            LOGGER.info("[SCIM] Valid mapping found, skipping");
                            continue;
                        } else {
                            LOGGER.info("[SCIM] Delete a dangling mapping");
                            getScimResourceDao().delete(mapping);
                        }
                    }

                    Optional<KeycloakId> mapped = tryToMap(resource);
                    if (mapped.isPresent()) {
                        LOGGER.info("[SCIM] Matched");
                        createMapping(mapped.get(), externalId);
                    } else {
                        switch (scimProviderConfiguration.getImportAction()) {
                            case CREATE_LOCAL:
                                LOGGER.info("[SCIM] Create local resource");
                                try {
                                    KeycloakId id = createEntity(resource);
                                    createMapping(id, externalId);
                                    syncRes.increaseAdded();
                                } catch (Exception e) {
                                    // TODO ExceptionHandling should we stop and throw ScimPropagationException here ?
                                    LOGGER.error(e);
                                }
                                break;
                            case DELETE_REMOTE:
                                LOGGER.info("[SCIM] Delete remote resource");
                                this.scimClient.delete(externalId);
                                syncRes.increaseRemoved();
                                break;
                            case NOTHING:
                                LOGGER.info("[SCIM] Import action set to NOTHING");
                                break;
                        }
                    }
                } catch (Exception e) {
                    // TODO ExceptionHandling should we stop and throw ScimPropagationException here ?
                    LOGGER.error(e);
                    e.printStackTrace();
                    syncRes.increaseFailed();
                }
            }
        } catch (ResponseException e) {
            // TODO ExceptionHandling should we stop and throw ScimPropagationException here ?
            LOGGER.error(e);
            e.printStackTrace();
            syncRes.increaseFailed();
        }
    }

    protected abstract KeycloakId createEntity(S resource) throws ScimPropagationException;

    protected abstract Optional<KeycloakId> tryToMap(S resource) throws ScimPropagationException;

    protected abstract boolean entityExists(KeycloakId keycloakId);

    public void sync(SynchronizationResult syncRes) throws ScimPropagationException {
        if (this.scimProviderConfiguration.isSyncImport()) {
            this.importResources(syncRes);
        }
        if (this.scimProviderConfiguration.isSyncRefresh()) {
            this.refreshResources(syncRes);
        }
    }

    protected Meta newMetaLocation(EntityOnRemoteScimId externalId) {
        Meta meta = new Meta();
        URI uri = getUri(type, externalId);
        meta.setLocation(uri.toString());
        return meta;
    }

    protected URI getUri(ScimResourceType type, EntityOnRemoteScimId externalId) {
        try {
            return new URI("%s/%s".formatted(type.getEndpoint(), externalId.asString()));
        } catch (URISyntaxException e) {
            throw new IllegalStateException("should never occur: can not format URI for type %s and id %s".formatted(type, externalId), e);
        }
    }

    protected KeycloakDao getKeycloakDao() {
        return new KeycloakDao(getKeycloakSession());
    }

    @Override
    public void close() {
        scimClient.close();
    }

    public ScrimProviderConfiguration getConfiguration() {
        return scimProviderConfiguration;
    }
}
