package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.common.resources.Group;
import de.captaingoldfish.scim.sdk.common.resources.complex.Meta;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.Member;
import jakarta.persistence.NoResultException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.jboss.logging.Logger;
import org.keycloak.models.GroupModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.UserModel;
import sh.libre.scim.jpa.ScimResource;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

public class GroupScimService extends AbstractScimService<GroupModel, Group> {
    private final Logger logger = Logger.getLogger(GroupScimService.class);

    public GroupScimService(KeycloakSession keycloakSession, ScrimProviderConfiguration scimProviderConfiguration) {
        super(keycloakSession, scimProviderConfiguration, ScimResourceType.GROUP);
    }

    @Override
    protected Stream<GroupModel> getResourceStream() {
        return getKeycloakDao().getGroupsStream();
    }

    @Override
    protected boolean entityExists(KeycloakId keycloakId) {
        return getKeycloakDao().groupExists(keycloakId);
    }

    @Override
    protected Optional<KeycloakId> tryToMap(Group resource) {
        String externalId = resource.getId().get();
        String displayName = resource.getDisplayName().get();
        Set<String> names = Set.of(externalId, displayName);
        Optional<GroupModel> group = getKeycloakDao().getGroupsStream()
                .filter(groupModel -> names.contains(groupModel.getName()))
                .findFirst();
        return group.map(GroupModel::getId).map(KeycloakId::new);
    }

    @Override
    protected KeycloakId createEntity(Group resource) {
        String displayName = resource.getDisplayName().get();
        GroupModel group = getKeycloakDao().createGroup(displayName);
        List<Member> groupMembers = resource.getMembers();
        if (CollectionUtils.isNotEmpty(groupMembers)) {
            for (Member groupMember : groupMembers) {
                try {
                    EntityOnRemoteScimId externalId = new EntityOnRemoteScimId(groupMember.getValue().get());
                    ScimResource userMapping = getScimResourceDao().findUserByExternalId(externalId).get();
                    KeycloakId userId = userMapping.getIdAsKeycloakId();
                    try {
                        UserModel user = getKeycloakDao().getUserById(userId);
                        if (user == null) {
                            throw new NoResultException();
                        }
                        user.joinGroup(group);
                    } catch (Exception e) {
                        logger.warn(e);
                    }
                } catch (NoSuchElementException e) {
                    logger.warnf("member %s not found for scim group %s", groupMember.getValue().get(), resource.getId().get());
                }
            }
        }
        return new KeycloakId(group.getId());
    }

    @Override
    protected boolean isSkip(GroupModel groupModel) {
        return BooleanUtils.TRUE.equals(groupModel.getFirstAttribute("scim-skip"));
    }

    @Override
    protected KeycloakId getId(GroupModel groupModel) {
        return new KeycloakId(groupModel.getId());
    }

    @Override
    protected Group toScimForCreation(GroupModel groupModel) {
        Set<KeycloakId> members = getKeycloakDao().getGroupMembers(groupModel);
        Group group = new Group();
        group.setExternalId(groupModel.getId());
        group.setDisplayName(groupModel.getName());
        for (KeycloakId member : members) {
            Member groupMember = new Member();
            try {
                ScimResource userMapping = getScimResourceDao().findUserById(member).get();
                logger.debug(userMapping.getExternalIdAsEntityOnRemoteScimId());
                logger.debug(userMapping.getIdAsKeycloakId());
                groupMember.setValue(userMapping.getExternalIdAsEntityOnRemoteScimId().asString());
                URI ref = new URI(String.format("Users/%s", userMapping.getExternalIdAsEntityOnRemoteScimId()));
                groupMember.setRef(ref.toString());
                group.addMember(groupMember);
            } catch (NoSuchElementException e) {
                logger.warnf("member %s not found for group %s", member, groupModel.getId());
            } catch (URISyntaxException e) {
                logger.warnf("bad ref uri");
            }
        }
        return group;
    }

    @Override
    protected Group toScimForReplace(GroupModel groupModel, EntityOnRemoteScimId externalId) {
        Group group = toScimForCreation(groupModel);
        group.setId(externalId.asString());
        Meta meta = newMetaLocation(externalId);
        group.setMeta(meta);
        return group;
    }

    @Override
    protected boolean isSkipRefresh(GroupModel resource) {
        return false;
    }
}
