package sh.libre.scim.core;

import org.jboss.logging.Logger;
import org.keycloak.component.ComponentModel;
import org.keycloak.models.KeycloakSession;
import sh.libre.scim.storage.ScimEndpointConfigurationStorageProviderFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * In charge of sending SCIM Request to all registered Scim endpoints.
 */
public class ScimDispatcher {

    private static final Logger logger = Logger.getLogger(ScimDispatcher.class);

    private final KeycloakSession session;
    private boolean clientsInitialized = false;
    private final List<UserScimService> userScimServices = new ArrayList<>();
    private final List<GroupScimService> groupScimServices = new ArrayList<>();


    public ScimDispatcher(KeycloakSession session) {
        this.session = session;
    }

    /**
     * Lists all active ScimStorageProviderFactory and create new ScimClients for each of them
     */
    public void refreshActiveScimEndpoints() {
        try {
            // Step 1: close existing clients
            for (GroupScimService c : groupScimServices) {
                c.close();
            }
            groupScimServices.clear();
            for (UserScimService c : userScimServices) {
                c.close();
            }
            userScimServices.clear();

            // Step 2: Get All SCIM endpoints defined in Admin Console (enabled ScimStorageProviderFactory)
            session.getContext().getRealm().getComponentsStream()
                    .filter(m -> ScimEndpointConfigurationStorageProviderFactory.ID.equals(m.getProviderId())
                                 && m.get("enabled", true))
                    .forEach(scimEndpointConfigurationRaw -> {
                        ScrimProviderConfiguration scrimProviderConfiguration = new ScrimProviderConfiguration(scimEndpointConfigurationRaw);
                        try {
                            // Step 3 : create scim clients for each endpoint
                            if (scimEndpointConfigurationRaw.get(ScrimProviderConfiguration.CONF_KEY_PROPAGATION_GROUP, false)) {
                                GroupScimService groupScimService = new GroupScimService(session, scrimProviderConfiguration);
                                groupScimServices.add(groupScimService);
                            }
                            if (scimEndpointConfigurationRaw.get(ScrimProviderConfiguration.CONF_KEY_PROPAGATION_USER, false)) {
                                UserScimService userScimService = new UserScimService(session, scrimProviderConfiguration);
                                userScimServices.add(userScimService);
                            }
                        } catch (Exception e) {
                            logger.warnf("[SCIM] Invalid Endpoint configuration %s : %s", scimEndpointConfigurationRaw.getId(), e.getMessage());
                            // TODO is it ok to log and try to create the other clients ?
                        }
                    });
        } catch (Exception e) {
            logger.error("[SCIM] Error while refreshing scim clients ", e);
            // TODO : how to handle exception here ?
        }
    }

    public void dispatchUserModificationToAll(Consumer<UserScimService> operationToDispatch) {
        initializeClientsIfNeeded();
        userScimServices.forEach(operationToDispatch);
        logger.infof("[SCIM] User operation dispatched to %d SCIM server", userScimServices.size());
    }

    public void dispatchGroupModificationToAll(Consumer<GroupScimService> operationToDispatch) {
        initializeClientsIfNeeded();
        groupScimServices.forEach(operationToDispatch);
        logger.infof("[SCIM] Group operation dispatched to %d SCIM server", groupScimServices.size());
    }

    public void dispatchUserModificationToOne(ComponentModel scimServerConfiguration, Consumer<UserScimService> operationToDispatch) {
        initializeClientsIfNeeded();
        // Scim client should already have been created
        Optional<UserScimService> matchingClient = userScimServices.stream().filter(u -> u.getConfiguration().getId().equals(scimServerConfiguration.getId())).findFirst();
        if (matchingClient.isPresent()) {
            operationToDispatch.accept(matchingClient.get());
            logger.infof("[SCIM] User operation dispatched to SCIM server %s", matchingClient.get().getConfiguration().getId());
        } else {
            logger.error("[SCIM] Could not find a Scim Client matching endpoint configuration" + scimServerConfiguration.getId());
        }
    }


    public void dispatchGroupModificationToOne(ComponentModel scimServerConfiguration, Consumer<GroupScimService> operationToDispatch) {
        initializeClientsIfNeeded();
        // Scim client should already have been created
        Optional<GroupScimService> matchingClient = groupScimServices.stream().filter(u -> u.getConfiguration().getId().equals(scimServerConfiguration.getId())).findFirst();
        if (matchingClient.isPresent()) {
            operationToDispatch.accept(matchingClient.get());
            logger.infof("[SCIM] Group operation dispatched to SCIM server %s", matchingClient.get().getConfiguration().getId());
        } else {
            logger.error("[SCIM] Could not find a Scim Client matching endpoint configuration" + scimServerConfiguration.getId());
        }
    }

    public void close() {
        for (GroupScimService c : groupScimServices) {
            c.close();
        }
        for (UserScimService c : userScimServices) {
            c.close();
        }
        groupScimServices.clear();
        userScimServices.clear();
    }

    private void initializeClientsIfNeeded() {
        if (!clientsInitialized) {
            clientsInitialized = true;
            refreshActiveScimEndpoints();
        }
    }
}
