package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.client.http.BasicAuth;
import org.keycloak.component.ComponentModel;

public class ScrimProviderConfiguration {

    private final String endPoint;
    private final String id;
    private final String contentType;
    private final String authorizationHeaderValue;
    private final ImportAction importAction;
    private final boolean syncImport;
    private final boolean syncRefresh;

    public ScrimProviderConfiguration(ComponentModel scimProviderConfiguration) {
        AuthMode authMode = AuthMode.valueOf(scimProviderConfiguration.get("auth-mode"));
        authorizationHeaderValue = switch (authMode) {
            case BEARER -> "Bearer " + scimProviderConfiguration.get("auth-pass");
            case BASIC_AUTH -> {
                BasicAuth basicAuth = BasicAuth.builder()
                        .username(scimProviderConfiguration.get("auth-user"))
                        .password(scimProviderConfiguration.get("auth-pass"))
                        .build();
                yield basicAuth.getAuthorizationHeaderValue();
            }
            default ->
                    throw new IllegalArgumentException("authMode " + scimProviderConfiguration + " is not supported");
        };
        contentType = scimProviderConfiguration.get("content-type");
        endPoint = scimProviderConfiguration.get("endpoint");
        id = scimProviderConfiguration.getId();
        importAction = ImportAction.valueOf(scimProviderConfiguration.get("sync-import-action"));
        syncImport = scimProviderConfiguration.get("sync-import", false);
        syncRefresh = scimProviderConfiguration.get("sync-refresh", false);
    }

    public boolean isSyncRefresh() {
        return syncRefresh;
    }

    public boolean isSyncImport() {
        return syncImport;
    }

    public String getContentType() {
        return contentType;
    }

    public String getAuthorizationHeaderValue() {
        return authorizationHeaderValue;
    }

    public String getId() {
        return id;
    }

    public ImportAction getImportAction() {
        return importAction;
    }

    public String getEndPoint() {
        return endPoint;
    }

    public enum AuthMode {
        BEARER, BASIC_AUTH, NONE
    }

    public enum EndpointContentType {
        JSON, SCIM_JSON
    }

    public enum ImportAction {
        CREATE_LOCAL, DELETE_REMOTE, NOTHING
    }
}
