package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.common.resources.Group;
import de.captaingoldfish.scim.sdk.common.resources.complex.Meta;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.Member;
import jakarta.persistence.NoResultException;
import org.apache.commons.lang3.BooleanUtils;
import org.jboss.logging.Logger;
import org.keycloak.models.GroupModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.UserModel;
import sh.libre.scim.jpa.ScimResource;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class GroupAdapter extends Adapter<GroupModel, Group> {

    private String displayName;
    private Set<String> members = new HashSet<>();

    public GroupAdapter(KeycloakSession session, String componentId) {
        super(session, componentId, "Group", Logger.getLogger(GroupAdapter.class));
    }

    public String getDisplayName() {
        return displayName;
    }

    public void setDisplayName(String displayName) {
        if (this.displayName == null) {
            this.displayName = displayName;
        }
    }

    @Override
    public Class<Group> getResourceClass() {
        return Group.class;
    }

    @Override
    public void apply(GroupModel group) {
        setId(group.getId());
        setDisplayName(group.getName());
        this.members = session.users()
                .getGroupMembersStream(session.getContext().getRealm(), group)
                .map(x -> x.getId())
                .collect(Collectors.toSet());
        this.skip = BooleanUtils.TRUE.equals(group.getFirstAttribute("scim-skip"));
    }

    @Override
    public void apply(Group group) {
        setExternalId(group.getId().get());
        setDisplayName(group.getDisplayName().get());
        List<Member> groupMembers = group.getMembers();
        if (groupMembers != null && groupMembers.size() > 0) {
            this.members = new HashSet<>();
            for (Member groupMember : groupMembers) {
                try {
                    ScimResource userMapping = this.query("findByExternalId", groupMember.getValue().get(), "User")
                            .getSingleResult();
                    this.members.add(userMapping.getId());
                } catch (NoResultException e) {
                    logger.warnf("member %s not found for scim group %s", groupMember.getValue().get(), group.getId().get());
                }
            }
        }
    }

    @Override
    public Group toSCIM() {
        Group group = new Group();
        group.setId(externalId);
        group.setExternalId(id);
        group.setDisplayName(displayName);
        if (members.size() > 0) {
            for (String member : members) {
                Member groupMember = new Member();
                try {
                    ScimResource userMapping = this.query("findById", member, "User").getSingleResult();
                    logger.debug(userMapping.getExternalId());
                    logger.debug(userMapping.getId());
                    groupMember.setValue(userMapping.getExternalId());
                    URI ref = new URI(String.format("Users/%s", userMapping.getExternalId()));
                    groupMember.setRef(ref.toString());
                    group.addMember(groupMember);
                } catch (NoResultException e) {
                    logger.warnf("member %s not found for group %s", member, id);
                } catch (URISyntaxException e) {
                    logger.warnf("bad ref uri");
                }
            }
        }
        Meta meta = new Meta();
        try {
            URI uri = new URI("Groups/" + externalId);
            meta.setLocation(uri.toString());
        } catch (URISyntaxException e) {
            logger.warn(e);
        }
        group.setMeta(meta);
        return group;
    }

    @Override
    public Boolean entityExists() {
        if (this.id == null) {
            return false;
        }
        GroupModel group = session.groups().getGroupById(realm, id);
        return group != null;
    }

    @Override
    public Boolean tryToMap() {
        Set<String> names = Set.of(externalId, displayName);
        Optional<GroupModel> group = session.groups().getGroupsStream(realm)
                .filter(groupModel -> names.contains(groupModel.getName()))
                .findFirst();
        if (group.isPresent()) {
            setId(group.get().getId());
            return true;
        }
        return false;
    }

    @Override
    public void createEntity() {
        GroupModel group = session.groups().createGroup(realm, displayName);
        this.id = group.getId();
        for (String mId : members) {
            try {
                UserModel user = session.users().getUserById(realm, mId);
                if (user == null) {
                    throw new NoResultException();
                }
                user.joinGroup(group);
            } catch (Exception e) {
                logger.warn(e);
            }
        }
    }

    @Override
    public Stream<GroupModel> getResourceStream() {
        return this.session.groups().getGroupsStream(this.session.getContext().getRealm());
    }

    @Override
    public Boolean skipRefresh() {
        return false;
    }

}
