package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.common.resources.User;
import de.captaingoldfish.scim.sdk.common.resources.complex.Meta;
import de.captaingoldfish.scim.sdk.common.resources.complex.Name;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.Email;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.PersonRole;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.jboss.logging.Logger;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.UserModel;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

public class UserAdapter extends Adapter<UserModel, User> {

    private String username;
    private String displayName;
    private String email;
    private Boolean active;
    private String[] roles;
    private String firstName;
    private String lastName;

    public UserAdapter(KeycloakSession session, String componentId) {
        super(session, componentId, "User", Logger.getLogger(UserAdapter.class));
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        if (this.username == null) {
            this.username = username;
        }
    }

    public String getDisplayName() {
        return displayName;
    }

    public void setDisplayName(String displayName) {
        if (this.displayName == null) {
            this.displayName = displayName;
        }
    }

    public String getEmail() {
        return email;
    }

    public void setEmail(String email) {
        if (this.email == null) {
            this.email = email;
        }
    }

    public Boolean getActive() {
        return active;
    }

    public void setActive(Boolean active) {
        if (this.active == null) {
            this.active = active;
        }
    }

    public String[] getRoles() {
        return roles;
    }

    public void setRoles(String[] roles) {
        this.roles = roles;
    }

    public String getFirstName() {
        return firstName;
    }

    public void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    public String getLastName() {
        return lastName;
    }

    public void setLastName(String lastName) {
        this.lastName = lastName;
    }

    @Override
    public Class<User> getResourceClass() {
        return User.class;
    }

    @Override
    public void apply(UserModel user) {
        setId(user.getId());
        setUsername(user.getUsername());
        String displayName = String.format("%s %s", StringUtils.defaultString(user.getFirstName()),
                StringUtils.defaultString(user.getLastName())).trim();
        if (StringUtils.isEmpty(displayName)) {
            displayName = user.getUsername();
        }
        setDisplayName(displayName);
        setEmail(user.getEmail());
        setActive(user.isEnabled());
        setFirstName(user.getFirstName());
        setLastName(user.getLastName());
        Set<String> rolesSet = new HashSet<>();
        user.getGroupsStream().flatMap(g -> g.getRoleMappingsStream())
                .filter((r) -> BooleanUtils.TRUE.equals(r.getFirstAttribute("scim")))
                .map((r) -> r.getName())
                .forEach(r -> rolesSet.add(r));
        user.getRoleMappingsStream()
                .filter((r) -> BooleanUtils.TRUE.equals(r.getFirstAttribute("scim")))
                .map((r) -> r.getName())
                .forEach(r -> rolesSet.add(r));
        String[] roles = new String[rolesSet.size()];
        rolesSet.toArray(roles);
        setRoles(roles);
        this.skip = BooleanUtils.TRUE.equals(user.getFirstAttribute("scim-skip"));
    }

    @Override
    public void apply(User user) {
        setExternalId(user.getId().get());
        setUsername(user.getUserName().get());
        setDisplayName(user.getDisplayName().get());
        setActive(user.isActive().get());
        if (user.getEmails().size() > 0) {
            setEmail(user.getEmails().get(0).getValue().get());
        }
    }

    @Override
    public User toSCIM() {
        User user = new User();
        user.setExternalId(id);
        user.setUserName(username);
        user.setId(externalId);
        user.setDisplayName(displayName);
        Name name = new Name();
        name.setFamilyName(lastName);
        name.setGivenName(firstName);
        user.setName(name);
        List<Email> emails = new ArrayList<>();
        if (email != null) {
            emails.add(
                    Email.builder().value(getEmail()).build());
        }
        user.setEmails(emails);
        user.setActive(active);
        Meta meta = new Meta();
        try {
            URI uri = new URI("Users/" + externalId);
            meta.setLocation(uri.toString());
        } catch (URISyntaxException e) {
            logger.warn(e);
        }
        user.setMeta(meta);
        List<PersonRole> roles = new ArrayList<>();
        for (String role : this.roles) {
            PersonRole personRole = new PersonRole();
            personRole.setValue(role);
            roles.add(personRole);
        }
        user.setRoles(roles);
        return user;
    }

    @Override
    public void createEntity() throws Exception {
        if (StringUtils.isEmpty(username)) {
            throw new Exception("can't create user with empty username");
        }
        UserModel user = session.users().addUser(realm, username);
        user.setEmail(email);
        user.setEnabled(active);
        this.id = user.getId();
    }

    @Override
    public Boolean entityExists() {
        if (this.id == null) {
            return false;
        }
        UserModel user = session.users().getUserById(realm, id);
        return user != null;
    }

    @Override
    public Boolean tryToMap() {
        UserModel sameUsernameUser = null;
        UserModel sameEmailUser = null;
        if (username != null) {
            sameUsernameUser = session.users().getUserByUsername(realm, username);
        }
        if (email != null) {
            sameEmailUser = session.users().getUserByEmail(realm, email);
        }
        if ((sameUsernameUser != null && sameEmailUser != null)
                && (!StringUtils.equals(sameUsernameUser.getId(), sameEmailUser.getId()))) {
            logger.warnf("found 2 possible users for remote user %s %s", username, email);
            return false;
        }
        if (sameUsernameUser != null) {
            this.id = sameUsernameUser.getId();
            return true;
        }
        if (sameEmailUser != null) {
            this.id = sameEmailUser.getId();
            return true;
        }
        return false;
    }

    @Override
    public Stream<UserModel> getResourceStream() {
        Map<String, String> params = new HashMap<>();
        return this.session.users().searchForUserStream(realm, params);
    }

    @Override
    public Boolean skipRefresh() {
        return "admin".equals(getUsername());
    }
}
