package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.common.exceptions.ResponseException;
import de.captaingoldfish.scim.sdk.common.resources.ResourceNode;
import de.captaingoldfish.scim.sdk.common.resources.complex.Meta;
import org.jboss.logging.Logger;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RoleMapperModel;
import org.keycloak.storage.user.SynchronizationResult;
import sh.libre.scim.jpa.ScimResource;
import sh.libre.scim.jpa.ScimResourceDao;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.stream.Stream;

public abstract class AbstractScimService<RMM extends RoleMapperModel, S extends ResourceNode> implements AutoCloseable {

    private static final Logger LOGGER = Logger.getLogger(AbstractScimService.class);

    private final KeycloakSession keycloakSession;

    private final ScrimProviderConfiguration scimProviderConfiguration;

    private final ScimResourceType type;

    private final ScimClient<S> scimClient;

    protected AbstractScimService(KeycloakSession keycloakSession, ScrimProviderConfiguration scimProviderConfiguration, ScimResourceType type) {
        this.keycloakSession = keycloakSession;
        this.scimProviderConfiguration = scimProviderConfiguration;
        this.type = type;
        this.scimClient = ScimClient.open(scimProviderConfiguration, type);
    }

    public void create(RMM roleMapperModel) {
        boolean skip = isSkip(roleMapperModel);
        if (skip)
            return;
        // If mapping exist then it was created by import so skip.
        KeycloakId id = getId(roleMapperModel);
        if (findById(id).isPresent()) {
            return;
        }
        ResourceNode scimForCreation = toScimForCreation(roleMapperModel);
        EntityOnRemoteScimId externalId = scimClient.create(scimForCreation);
        createMapping(id, externalId);
    }

    protected abstract ResourceNode toScimForCreation(RMM roleMapperModel);

    protected abstract KeycloakId getId(RMM roleMapperModel);

    protected abstract boolean isSkip(RMM roleMapperModel);

    private void createMapping(KeycloakId keycloakId, EntityOnRemoteScimId externalId) {
        getScimResourceDao().create(keycloakId, externalId, type);
    }

    protected ScimResourceDao getScimResourceDao() {
        return ScimResourceDao.newInstance(getKeycloakSession(), scimProviderConfiguration.getId());
    }

    private Optional<ScimResource> findById(KeycloakId keycloakId) {
        return getScimResourceDao().findById(keycloakId, type);
    }

    private KeycloakSession getKeycloakSession() {
        return keycloakSession;
    }

    public void replace(RMM roleMapperModel) {
        try {
            if (isSkip(roleMapperModel))
                return;
            KeycloakId id = getId(roleMapperModel);
            ScimResource scimResource = findById(id).get();
            EntityOnRemoteScimId externalId = scimResource.getExternalIdAsEntityOnRemoteScimId();
            ResourceNode scimForReplace = toScimForReplace(roleMapperModel, externalId);
            scimClient.replace(externalId, scimForReplace);
        } catch (NoSuchElementException e) {
            LOGGER.warnf("failed to replace resource %s, scim mapping not found", getId(roleMapperModel));
        } catch (Exception e) {
            LOGGER.error(e);
        }
    }

    protected abstract ResourceNode toScimForReplace(RMM roleMapperModel, EntityOnRemoteScimId externalId);

    public void delete(KeycloakId id) {
        try {
            ScimResource resource = findById(id).get();
            EntityOnRemoteScimId externalId = resource.getExternalIdAsEntityOnRemoteScimId();
            scimClient.delete(externalId);
            getScimResourceDao().delete(resource);
        } catch (NoSuchElementException e) {
            LOGGER.warnf("Failed to delete resource %s, scim mapping not found", id);
        }
    }

    public void refreshResources(SynchronizationResult syncRes) {
        LOGGER.info("Refresh resources");
        getResourceStream().forEach(resource -> {
            KeycloakId id = getId(resource);
            LOGGER.infof("Reconciling local resource %s", id);
            if (!isSkipRefresh(resource)) {
                try {
                    findById(id).get();
                    LOGGER.info("Replacing it");
                    replace(resource);
                } catch (NoSuchElementException e) {
                    LOGGER.info("Creating it");
                    create(resource);
                }
                syncRes.increaseUpdated();
            }
        });
    }

    protected abstract boolean isSkipRefresh(RMM resource);

    protected abstract Stream<RMM> getResourceStream();

    public void importResources(SynchronizationResult syncRes) {
        LOGGER.info("Import");
        ScimClient<S> scimClient = this.scimClient;
        try {
            for (S resource : scimClient.listResources()) {
                try {
                    LOGGER.infof("Reconciling remote resource %s", resource);
                    EntityOnRemoteScimId externalId = resource.getId().map(EntityOnRemoteScimId::new).get();
                    try {
                        ScimResource mapping = getScimResourceDao().findByExternalId(externalId, type).get();
                        if (entityExists(mapping.getIdAsKeycloakId())) {
                            LOGGER.info("Valid mapping found, skipping");
                            continue;
                        } else {
                            LOGGER.info("Delete a dangling mapping");
                            getScimResourceDao().delete(mapping);
                        }
                    } catch (NoSuchElementException e) {
                        // nothing to do
                    }

                    Optional<KeycloakId> mapped = tryToMap(resource);
                    if (mapped.isPresent()) {
                        LOGGER.info("Matched");
                        createMapping(mapped.get(), externalId);
                    } else {
                        switch (scimProviderConfiguration.getImportAction()) {
                            case CREATE_LOCAL:
                                LOGGER.info("Create local resource");
                                try {
                                    KeycloakId id = createEntity(resource);
                                    createMapping(id, externalId);
                                    syncRes.increaseAdded();
                                } catch (Exception e) {
                                    LOGGER.error(e);
                                }
                                break;
                            case DELETE_REMOTE:
                                LOGGER.info("Delete remote resource");
                                scimClient.delete(externalId);
                                syncRes.increaseRemoved();
                                break;
                            case NOTHING:
                                LOGGER.info("Import action set to NOTHING");
                                break;
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error(e);
                    e.printStackTrace();
                    syncRes.increaseFailed();
                }
            }
        } catch (ResponseException e) {
            throw new RuntimeException(e);
        }
    }

    protected abstract KeycloakId createEntity(S resource);

    protected abstract Optional<KeycloakId> tryToMap(S resource);

    protected abstract boolean entityExists(KeycloakId keycloakId);

    public void sync(SynchronizationResult syncRes) {
        if (this.scimProviderConfiguration.isSyncImport()) {
            this.importResources(syncRes);
        }
        if (this.scimProviderConfiguration.isSyncRefresh()) {
            this.refreshResources(syncRes);
        }
    }

    protected Meta newMetaLocation(EntityOnRemoteScimId externalId) {
        Meta meta = new Meta();
        try {
            URI uri = new URI("%ss/%s".formatted(type, externalId.asString()));
            meta.setLocation(uri.toString());
        } catch (URISyntaxException e) {
            LOGGER.warn(e);
        }
        return meta;
    }

    protected KeycloakDao getKeycloakDao() {
        return new KeycloakDao(getKeycloakSession());
    }

    @Override
    public void close() {
        scimClient.close();
    }

    public ScrimProviderConfiguration getConfiguration() {
        return scimProviderConfiguration;
    }
}
