package sh.libre.scim.event;

import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.jboss.logging.Logger;
import org.keycloak.events.Event;
import org.keycloak.events.EventListenerProvider;
import org.keycloak.events.EventType;
import org.keycloak.events.admin.AdminEvent;
import org.keycloak.events.admin.OperationType;
import org.keycloak.events.admin.ResourceType;
import org.keycloak.models.GroupModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.UserModel;
import sh.libre.scim.core.GroupAdapter;
import sh.libre.scim.core.ScimDispatcher;
import sh.libre.scim.core.UserAdapter;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class ScimEventListenerProvider implements EventListenerProvider {

    private static final Logger LOGGER = Logger.getLogger(ScimEventListenerProvider.class);

    private final ScimDispatcher dispatcher;

    private final KeycloakSession session;

    private final Map<ResourceType, Pattern> patterns = new HashMap<>();

    public ScimEventListenerProvider(KeycloakSession session) {
        this.session = session;
        dispatcher = new ScimDispatcher(session);
        patterns.put(ResourceType.USER, Pattern.compile("users/(.+)"));
        patterns.put(ResourceType.GROUP, Pattern.compile("groups/([\\w-]+)(/children)?"));
        patterns.put(ResourceType.GROUP_MEMBERSHIP, Pattern.compile("users/(.+)/groups/(.+)"));
        patterns.put(ResourceType.REALM_ROLE_MAPPING, Pattern.compile("^(.+)/(.+)/role-mappings"));
    }

    @Override
    public void close() {
    }

    @Override
    public void onEvent(Event event) {
        EventType eventType = event.getType();
        String eventUserId = event.getUserId();
        switch (eventType) {
            case REGISTER -> {
                UserModel user = getUser(eventUserId);
                dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.create(UserAdapter.class, user));
            }
            case UPDATE_EMAIL, UPDATE_PROFILE -> {
                UserModel user = getUser(eventUserId);
                dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.replace(UserAdapter.class, user));
            }
            case DELETE_ACCOUNT ->
                    dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.delete(UserAdapter.class, eventUserId));
            default -> LOGGER.trace("ignore event " + eventType);
        }
    }

    @Override
    public void onEvent(AdminEvent event, boolean includeRepresentation) {
        Pattern pattern = patterns.get(event.getResourceType());
        if (pattern == null)
            return;
        Matcher matcher = pattern.matcher(event.getResourcePath());
        if (!matcher.find())
            return;
        switch (event.getResourceType()) {
            case USER -> {
                String userId = matcher.group(1);
                LOGGER.infof("%s %s", userId, event.getOperationType());
                switch (event.getOperationType()) {
                    case CREATE -> {
                        UserModel user = getUser(userId);
                        dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.create(UserAdapter.class, user));
                        user.getGroupsStream().forEach(group -> {
                            dispatcher.run(ScimDispatcher.SCOPE_GROUP, (client) -> client.replace(GroupAdapter.class, group));
                        });
                    }
                    case UPDATE -> {
                        UserModel user = getUser(userId);
                        dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.replace(UserAdapter.class, user));
                    }
                    case DELETE ->
                            dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.delete(UserAdapter.class, userId));
                }
            }
            case GROUP -> {
                String groupId = matcher.group(1);
                LOGGER.infof("group %s %s", groupId, event.getOperationType());
                switch (event.getOperationType()) {
                    case CREATE -> {
                        GroupModel group = getGroup(groupId);
                        dispatcher.run(ScimDispatcher.SCOPE_GROUP, (client) -> client.create(GroupAdapter.class, group));
                    }
                    case UPDATE -> {
                        GroupModel group = getGroup(groupId);
                        dispatcher.run(ScimDispatcher.SCOPE_GROUP, (client) -> client.replace(GroupAdapter.class, group));
                    }
                    case DELETE -> dispatcher.run(ScimDispatcher.SCOPE_GROUP,
                            (client) -> client.delete(GroupAdapter.class, groupId));
                }
            }
            case GROUP_MEMBERSHIP -> {
                String userId = matcher.group(1);
                String groupId = matcher.group(2);
                LOGGER.infof("%s %s from %s", event.getOperationType(), userId, groupId);
                GroupModel group = getGroup(groupId);
                group.setSingleAttribute("scim-dirty", BooleanUtils.TRUE);
                UserModel user = getUser(userId);
                dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.replace(UserAdapter.class, user));
            }
            case REALM_ROLE_MAPPING -> {
                String type = matcher.group(1);
                String id = matcher.group(2);
                LOGGER.infof("%s %s %s roles", event.getOperationType(), type, id);
                switch (type) {
                    case "users" -> {
                        UserModel user = getUser(id);
                        dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.replace(UserAdapter.class, user));
                    }
                    case "groups" -> {
                        GroupModel group = getGroup(id);
                        session.users().getGroupMembersStream(session.getContext().getRealm(), group).forEach(user -> {
                            dispatcher.run(ScimDispatcher.SCOPE_USER, (client) -> client.replace(UserAdapter.class, user));
                        });
                    }
                }
            }
        }
    }

    private UserModel getUser(String id) {
        return session.users().getUserById(session.getContext().getRealm(), id);
    }

    private GroupModel getGroup(String id) {
        return session.groups().getGroupById(session.getContext().getRealm(), id);
    }
}
