package sh.libre.scim.core;

import com.google.common.net.HttpHeaders;
import de.captaingoldfish.scim.sdk.client.ScimClientConfig;
import de.captaingoldfish.scim.sdk.client.ScimRequestBuilder;
import de.captaingoldfish.scim.sdk.client.response.ServerResponse;
import de.captaingoldfish.scim.sdk.common.exceptions.ResponseException;
import de.captaingoldfish.scim.sdk.common.resources.ResourceNode;
import de.captaingoldfish.scim.sdk.common.response.ListResponse;
import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.github.resilience4j.retry.RetryRegistry;
import jakarta.ws.rs.ProcessingException;
import org.jboss.logging.Logger;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ScimClient<S extends ResourceNode> implements AutoCloseable {
    private final Logger logger = Logger.getLogger(ScimClient.class);

    private final RetryRegistry retryRegistry;

    private final ScimRequestBuilder scimRequestBuilder;

    private final ScimResourceType scimResourceType;

    {
        RetryConfig retryConfig = RetryConfig.custom()
                .maxAttempts(10)
                .intervalFunction(IntervalFunction.ofExponentialBackoff())
                .retryExceptions(ProcessingException.class)
                .build();
        retryRegistry = RetryRegistry.of(retryConfig);
    }

    private ScimClient(ScimRequestBuilder scimRequestBuilder, ScimResourceType scimResourceType) {
        this.scimRequestBuilder = scimRequestBuilder;
        this.scimResourceType = scimResourceType;
    }

    public static ScimClient open(ScrimProviderConfiguration scimProviderConfiguration, ScimResourceType scimResourceType) {
        String scimApplicationBaseUrl = scimProviderConfiguration.getEndPoint();
        Map<String, String> httpHeaders = new HashMap<>();
        httpHeaders.put(HttpHeaders.AUTHORIZATION, scimProviderConfiguration.getAuthorizationHeaderValue());
        httpHeaders.put(HttpHeaders.CONTENT_TYPE, scimProviderConfiguration.getContentType());
        ScimClientConfig scimClientConfig = ScimClientConfig.builder()
                .httpHeaders(httpHeaders)
                .connectTimeout(5)
                .requestTimeout(5)
                .socketTimeout(5)
                .expectedHttpResponseHeaders(Collections.emptyMap()) // strange, useful?
                // TODO Question Indiehoster : should we really allow connection with TLS ? .hostnameVerifier((s, sslSession) -> true)
                .build();
        ScimRequestBuilder scimRequestBuilder =
                new ScimRequestBuilder(
                        scimApplicationBaseUrl,
                        scimClientConfig
                );
        return new ScimClient(scimRequestBuilder, scimResourceType);
    }

    public EntityOnRemoteScimId create(KeycloakId id, ResourceNode scimForCreation) {
        if (scimForCreation.getId().isPresent()) {
            throw new IllegalArgumentException(
                    "%s is already created on remote with id %s".formatted(id, scimForCreation.getId().get())
            );
        }
        Retry retry = retryRegistry.retry("create-%s".formatted(id.asString()));
        ServerResponse<S> response = retry.executeSupplier(() -> {
            try {
                return scimRequestBuilder
                        .create(getResourceClass(), getScimEndpoint())
                        .setResource(scimForCreation)
                        .sendRequest();
            } catch (ResponseException e) {
                throw new RuntimeException(e);
            }
        });
        checkResponseIsSuccess(response);
        S resource = response.getResource();
        return resource.getId()
                .map(EntityOnRemoteScimId::new)
                .orElseThrow(() -> new IllegalStateException("created resource does not have id"));
    }

    private void checkResponseIsSuccess(ServerResponse<S> response) {
        if (!response.isSuccess()) {
            logger.warn(response.getResponseBody());
            logger.warn(response.getHttpStatus());
        }
    }

    private String getScimEndpoint() {
        return scimResourceType.getEndpoint();
    }

    private Class<S> getResourceClass() {
        return scimResourceType.getResourceClass();
    }

    public void replace(EntityOnRemoteScimId externalId, ResourceNode scimForReplace) {
        Retry retry = retryRegistry.retry("replace-%s".formatted(externalId.asString()));
        ServerResponse<S> response = retry.executeSupplier(() -> {
            try {
                return scimRequestBuilder
                        .update(getResourceClass(), getScimEndpoint(), externalId.asString())
                        .setResource(scimForReplace)
                        .sendRequest();
            } catch (ResponseException e) {
                throw new RuntimeException(e);
            }
        });
        checkResponseIsSuccess(response);
    }

    public void delete(EntityOnRemoteScimId externalId) {
        Retry retry = retryRegistry.retry("delete-%s".formatted(externalId.asString()));
        ServerResponse<S> response = retry.executeSupplier(() -> {
            try {
                return scimRequestBuilder.delete(getResourceClass(), getScimEndpoint(), externalId.asString())
                        .sendRequest();
            } catch (ResponseException e) {
                throw new RuntimeException(e);
            }
        });
        checkResponseIsSuccess(response);
    }

    @Override
    public void close() {
        scimRequestBuilder.close();
    }

    public List<S> listResources() {
        ServerResponse<ListResponse<S>> response = scimRequestBuilder.list(getResourceClass(), getScimEndpoint()).get().sendRequest();
        ListResponse<S> resourceTypeListResponse = response.getResource();
        return resourceTypeListResponse.getListedResources();
    }
}
