package sh.libre.scim.core;

import de.captaingoldfish.scim.sdk.common.resources.User;
import de.captaingoldfish.scim.sdk.common.resources.complex.Meta;
import de.captaingoldfish.scim.sdk.common.resources.complex.Name;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.Email;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.MultiComplexNode;
import de.captaingoldfish.scim.sdk.common.resources.multicomplex.PersonRole;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.jboss.logging.Logger;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RoleMapperModel;
import org.keycloak.models.RoleModel;
import org.keycloak.models.UserModel;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Stream;

public class UserScimService extends AbstractScimService<UserModel, User> {
    private final Logger logger = Logger.getLogger(UserScimService.class);

    public UserScimService(
            KeycloakSession keycloakSession,
            ScrimProviderConfiguration scimProviderConfiguration) {
        super(keycloakSession, scimProviderConfiguration, ScimResourceType.USER);
    }

    @Override
    protected Stream<UserModel> getResourceStream() {
        return getKeycloakDao().getUsersStream();
    }

    @Override
    protected boolean entityExists(KeycloakId keycloakId) {
        return getKeycloakDao().userExists(keycloakId);
    }

    @Override
    protected Optional<KeycloakId> tryToMap(User resource) {
        String username = resource.getUserName().get();
        String email = resource.getEmails().stream()
                .findFirst()
                .flatMap(MultiComplexNode::getValue)
                .orElse(null);
        UserModel sameUsernameUser = null;
        UserModel sameEmailUser = null;
        if (username != null) {
            sameUsernameUser = getKeycloakDao().getUserByUsername(username);
        }
        if (email != null) {
            sameEmailUser = getKeycloakDao().getUserByEmail(email);
        }
        if ((sameUsernameUser != null && sameEmailUser != null)
            && (!StringUtils.equals(sameUsernameUser.getId(), sameEmailUser.getId()))) {
            logger.warnf("found 2 possible users for remote user %s %s", username, email);
            return Optional.empty();
        }
        if (sameUsernameUser != null) {
            return Optional.of(getId(sameUsernameUser));
        }
        if (sameEmailUser != null) {
            return Optional.of(getId(sameEmailUser));
        }
        return Optional.empty();
    }

    @Override
    protected KeycloakId createEntity(User resource) {
        String username = resource.getUserName().get();
        if (StringUtils.isEmpty(username)) {
            throw new IllegalArgumentException("can't create user with empty username");
        }
        UserModel user = getKeycloakDao().addUser(username);
        resource.getEmails().stream()
                .findFirst()
                .flatMap(MultiComplexNode::getValue)
                .ifPresent(user::setEmail);
        user.setEnabled(resource.isActive().get());
        return new KeycloakId(user.getId());
    }

    @Override
    protected boolean isSkip(UserModel userModel) {
        return BooleanUtils.TRUE.equals(userModel.getFirstAttribute("scim-skip"));
    }

    @Override
    protected KeycloakId getId(UserModel userModel) {
        return new KeycloakId(userModel.getId());
    }

    @Override
    protected User toScimForCreation(UserModel roleMapperModel) {
        String firstAndLastName = String.format("%s %s",
                StringUtils.defaultString(roleMapperModel.getFirstName()),
                StringUtils.defaultString(roleMapperModel.getLastName())).trim();
        String displayName = StringUtils.defaultString(firstAndLastName, roleMapperModel.getUsername());
        Stream<RoleModel> groupRoleModels = roleMapperModel.getGroupsStream().flatMap(RoleMapperModel::getRoleMappingsStream);
        Stream<RoleModel> roleModels = roleMapperModel.getRoleMappingsStream();
        Stream<RoleModel> allRoleModels = Stream.concat(groupRoleModels, roleModels);
        List<PersonRole> roles = allRoleModels
                .filter((r) -> BooleanUtils.TRUE.equals(r.getFirstAttribute("scim")))
                .map(RoleModel::getName)
                .map(roleName -> {
                    PersonRole personRole = new PersonRole();
                    personRole.setValue(roleName);
                    return personRole;
                })
                .toList();
        User user = new User();
        user.setRoles(roles);
        user.setExternalId(roleMapperModel.getId());
        user.setUserName(roleMapperModel.getUsername());
        user.setDisplayName(displayName);
        Name name = new Name();
        name.setFamilyName(roleMapperModel.getLastName());
        name.setGivenName(roleMapperModel.getFirstName());
        user.setName(name);
        List<Email> emails = new ArrayList<>();
        if (roleMapperModel.getEmail() != null) {
            emails.add(
                    Email.builder().value(roleMapperModel.getEmail()).build());
        }
        user.setEmails(emails);
        user.setActive(roleMapperModel.isEnabled());
        return user;
    }

    @Override
    protected User toScimForReplace(UserModel userModel, EntityOnRemoteScimId externalId) {
        User user = toScimForCreation(userModel);
        user.setId(externalId.asString());
        Meta meta = newMetaLocation(externalId);
        user.setMeta(meta);
        return user;
    }

    @Override
    protected boolean isSkipRefresh(UserModel userModel) {
        return "admin".equals(userModel.getUsername());
    }
}
