package sh.libre.scim.core;

import com.google.common.net.HttpHeaders;
import de.captaingoldfish.scim.sdk.client.ScimClientConfig;
import de.captaingoldfish.scim.sdk.client.ScimRequestBuilder;
import de.captaingoldfish.scim.sdk.client.http.BasicAuth;
import de.captaingoldfish.scim.sdk.client.response.ServerResponse;
import de.captaingoldfish.scim.sdk.common.exceptions.ResponseException;
import de.captaingoldfish.scim.sdk.common.resources.ResourceNode;
import de.captaingoldfish.scim.sdk.common.response.ListResponse;
import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import io.github.resilience4j.retry.RetryRegistry;
import jakarta.persistence.EntityManager;
import jakarta.persistence.NoResultException;
import jakarta.ws.rs.ProcessingException;
import org.jboss.logging.Logger;
import org.keycloak.component.ComponentModel;
import org.keycloak.connections.jpa.JpaConnectionProvider;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RoleMapperModel;
import org.keycloak.storage.user.SynchronizationResult;
import sh.libre.scim.jpa.ScimResource;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;


public class ScimClient implements AutoCloseable {

    private static final Logger LOGGER = Logger.getLogger(ScimClient.class);

    private final ScimRequestBuilder scimRequestBuilder;

    private final RetryRegistry registry;

    private final KeycloakSession session;

    private final ComponentModel model;

    private ScimClient(ScimRequestBuilder scimRequestBuilder, RetryRegistry registry, KeycloakSession session, ComponentModel model) {
        this.scimRequestBuilder = scimRequestBuilder;
        this.registry = registry;
        this.session = session;
        this.model = model;
    }

    public static ScimClient newScimClient(ComponentModel model, KeycloakSession session) {
        String authMode = model.get("auth-mode");
        String authorizationHeaderValue = switch (authMode) {
            case "BEARER" -> "Bearer " + model.get("auth-pass");
            case "BASIC_AUTH" -> {
                BasicAuth basicAuth = BasicAuth.builder()
                        .username(model.get("auth-user"))
                        .password(model.get("auth-pass"))
                        .build();
                yield basicAuth.getAuthorizationHeaderValue();
            }
            default -> throw new IllegalArgumentException("authMode " + authMode + " is not supported");
        };

        Map<String, String> httpHeaders = new HashMap<>();
        httpHeaders.put(HttpHeaders.AUTHORIZATION, authorizationHeaderValue);
        httpHeaders.put(HttpHeaders.CONTENT_TYPE, model.get("content-type"));

        ScimClientConfig scimClientConfig = ScimClientConfig.builder()
                .httpHeaders(httpHeaders)
                .connectTimeout(5)
                .requestTimeout(5)
                .socketTimeout(5)
                .expectedHttpResponseHeaders(Collections.emptyMap()) // strange, useful?
                .hostnameVerifier((s, sslSession) -> true)
                .build();

        String scimApplicationBaseUrl = model.get("endpoint");
        ScimRequestBuilder scimRequestBuilder =
                new ScimRequestBuilder(
                        scimApplicationBaseUrl,
                        scimClientConfig
                );

        RetryConfig retryConfig = RetryConfig.custom()
                .maxAttempts(10)
                .intervalFunction(IntervalFunction.ofExponentialBackoff())
                .retryExceptions(ProcessingException.class)
                .build();

        RetryRegistry retryRegistry = RetryRegistry.of(retryConfig);
        return new ScimClient(scimRequestBuilder, retryRegistry, session, model);
    }

    protected EntityManager getEntityManager() {
        return session.getProvider(JpaConnectionProvider.class).getEntityManager();
    }

    protected String getRealmId() {
        return session.getContext().getRealm().getId();
    }

    protected <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> A newAdapter(
            Class<A> adapterClass) {
        try {
            return adapterClass.getDeclaredConstructor(KeycloakSession.class, String.class)
                    .newInstance(session, this.model.getId());
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void create(Class<A> adapterClass,
                                                                                                    M kcModel) {
        A adapter = newAdapter(adapterClass);
        adapter.apply(kcModel);
        if (adapter.skip)
            return;
        // If mapping exist then it was created by import so skip.
        if (!adapter.query("findById", adapter.getId()).getResultList().isEmpty()) {
            return;
        }
        Retry retry = registry.retry("create-" + adapter.getId());

        ServerResponse<S> response = retry.executeSupplier(() -> {
            try {
                return scimRequestBuilder
                        .create(adapter.getResourceClass(), adapter.getScimEndpoint())
                        .setResource(adapter.toScim())
                        .sendRequest();
            } catch (ResponseException e) {
                throw new RuntimeException(e);
            }
        });

        if (!response.isSuccess()) {
            LOGGER.warn(response.getResponseBody());
            LOGGER.warn(response.getHttpStatus());
        }

        adapter.apply(response.getResource());
        adapter.saveMapping();
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void replace(Class<A> adapterClass,
                                                                                                     M kcModel) {
        A adapter = newAdapter(adapterClass);
        try {
            adapter.apply(kcModel);
            if (adapter.skip)
                return;
            ScimResource resource = adapter.query("findById", adapter.getId()).getSingleResult();
            adapter.apply(resource);
            Retry retry = registry.retry("replace-" + adapter.getId());
            ServerResponse<S> response = retry.executeSupplier(() -> {
                try {
                    return scimRequestBuilder
                            .update(adapter.getResourceClass(), adapter.getScimEndpoint(), adapter.getExternalId())
                            .setResource(adapter.toScim())
                            .sendRequest();
                } catch (ResponseException e) {
                    throw new RuntimeException(e);
                }
            });
            if (!response.isSuccess()) {
                LOGGER.warn(response.getResponseBody());
                LOGGER.warn(response.getHttpStatus());
            }
        } catch (NoResultException e) {
            LOGGER.warnf("failed to replace resource %s, scim mapping not found", adapter.getId());
        } catch (Exception e) {
            LOGGER.error(e);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void delete(Class<A> adapterClass,
                                                                                                    String id) {
        A adapter = newAdapter(adapterClass);
        adapter.setId(id);

        try {
            ScimResource resource = adapter.query("findById", adapter.getId()).getSingleResult();
            adapter.apply(resource);

            Retry retry = registry.retry("delete-" + id);

            ServerResponse<S> response = retry.executeSupplier(() -> {
                try {
                    return scimRequestBuilder.delete(adapter.getResourceClass(), adapter.getScimEndpoint(), adapter.getExternalId())
                            .sendRequest();
                } catch (ResponseException e) {
                    throw new RuntimeException(e);
                }
            });

            if (!response.isSuccess()) {
                LOGGER.warn(response.getResponseBody());
                LOGGER.warn(response.getHttpStatus());
            }

            getEntityManager().remove(resource);

        } catch (NoResultException e) {
            LOGGER.warnf("Failed to delete resource %s, scim mapping not found", id);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void refreshResources(
            Class<A> adapterClass,
            SynchronizationResult syncRes) {
        LOGGER.info("Refresh resources");
        newAdapter(adapterClass).getResourceStream().forEach(resource -> {
            A adapter = newAdapter(adapterClass);
            adapter.apply(resource);
            LOGGER.infof("Reconciling local resource %s", adapter.getId());
            if (!adapter.skipRefresh()) {
                ScimResource mapping = adapter.getMapping();
                if (mapping == null) {
                    LOGGER.info("Creating it");
                    this.create(adapterClass, resource);
                } else {
                    LOGGER.info("Replacing it");
                    this.replace(adapterClass, resource);
                }
                syncRes.increaseUpdated();
            }
        });

    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void importResources(
            Class<A> adapterClass, SynchronizationResult syncRes) {
        LOGGER.info("Import");
        try {
            A adapter = newAdapter(adapterClass);
            ServerResponse<ListResponse<S>> response = scimRequestBuilder.list(adapter.getResourceClass(), adapter.getScimEndpoint()).get().sendRequest();
            ListResponse<S> resourceTypeListResponse = response.getResource();

            for (S resource : resourceTypeListResponse.getListedResources()) {
                try {
                    LOGGER.infof("Reconciling remote resource %s", resource);
                    adapter = newAdapter(adapterClass);
                    adapter.apply(resource);

                    ScimResource mapping = adapter.getMapping();
                    if (mapping != null) {
                        adapter.apply(mapping);
                        if (adapter.entityExists()) {
                            LOGGER.info("Valid mapping found, skipping");
                            continue;
                        } else {
                            LOGGER.info("Delete a dangling mapping");
                            adapter.deleteMapping();
                        }
                    }

                    Boolean mapped = adapter.tryToMap();
                    if (mapped) {
                        LOGGER.info("Matched");
                        adapter.saveMapping();
                    } else {
                        switch (this.model.get("sync-import-action")) {
                            case "CREATE_LOCAL":
                                LOGGER.info("Create local resource");
                                try {
                                    adapter.createEntity();
                                    adapter.saveMapping();
                                    syncRes.increaseAdded();
                                } catch (Exception e) {
                                    LOGGER.error(e);
                                }
                                break;
                            case "DELETE_REMOTE":
                                LOGGER.info("Delete remote resource");
                                scimRequestBuilder
                                        .delete(adapter.getResourceClass(), adapter.getScimEndpoint(), resource.getId().get())
                                        .sendRequest();
                                syncRes.increaseRemoved();
                                break;
                        }
                    }
                } catch (Exception e) {
                    LOGGER.error(e);
                    e.printStackTrace();
                    syncRes.increaseFailed();
                }
            }
        } catch (ResponseException e) {
            throw new RuntimeException(e);
        }
    }

    public <M extends RoleMapperModel, S extends ResourceNode, A extends Adapter<M, S>> void sync(Class<A> aClass,
                                                                                                  SynchronizationResult syncRes) {
        if (this.model.get("sync-import", false)) {
            this.importResources(aClass, syncRes);
        }
        if (this.model.get("sync-refresh", false)) {
            this.refreshResources(aClass, syncRes);
        }
    }

    @Override
    public void close() {
        scimRequestBuilder.close();
    }
}
