"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScimEndpoint = void 0;
const accessors_1 = require("@rocket.chat/apps-engine/definition/accessors");
const api_1 = require("@rocket.chat/apps-engine/definition/api");
const EmptyRequestError_1 = require("../errors/EmptyRequestError");
const EmptyResponseError_1 = require("../errors/EmptyResponseError");
const JsonParseError_1 = require("../errors/JsonParseError");
const Error_1 = require("../scim/Error");
const endpoint_1 = require("./common/endpoint");
const Context_1 = require("./Context");
class ScimEndpoint extends api_1.ApiEndpoint {
    constructor(app) {
        super(app);
        this.handleError = endpoint_1.handleRcError;
        this.get = this.wrapMethod("get");
        this.post = this.wrapMethod("post");
        this.put = this.wrapMethod("put");
        this.delete = this.wrapMethod("delete");
    }
    success(content) {
        return this.response({
            status: accessors_1.HttpStatusCode.OK,
            content,
        });
    }
    response(response) {
        if (response.headers === undefined) {
            response.headers = {};
        }
        response.headers["Content-Type"] = "application/scim+json";
        return response;
    }
    error(error) {
        return this.response({
            status: parseInt(error.status, 10),
            content: error,
        });
    }
    parseResponse(response) {
        if (!response.content) {
            throw new EmptyResponseError_1.EmptyResponseError();
        }
        let content;
        try {
            content = JSON.parse(response.content);
        }
        catch (e) {
            throw new JsonParseError_1.JsonParseError();
        }
        return content;
    }
    hasContent(request) {
        if (!request.content || Object.keys(request.content).length === 0) {
            throw new EmptyRequestError_1.EmptyRequestError();
        }
    }
    wrapMethod(name) {
        const method = this[`_${name}`];
        if (method === undefined || typeof method !== "function") {
            return undefined;
        }
        return async (request, endpoint, read, modify, http, persis) => {
            const logger = this.app.getLogger();
            try {
                const ctx = new Context_1.Context(request, endpoint, read, modify, http, persis, logger);
                logger.debug(`SCIM Request ${name.toUpperCase()} /${this.path}`, request.content);
                const response = await method.bind(this)(ctx);
                logger.debug("SCIM Response", response);
                return response;
            }
            catch (e) {
                let err;
                if (e.toSCIMError && typeof e.toSCIMError === "function") {
                    err = e.toSCIMError();
                }
                else {
                    err = new Error_1.SCIMError()
                        .setStatus(accessors_1.HttpStatusCode.INTERNAL_SERVER_ERROR)
                        .setScimType(Error_1.SCIMErrorType.INVALID_VALUE)
                        .setDetail(e.message);
                }
                logger.error(e);
                return this.error(err);
            }
        };
    }
}
exports.ScimEndpoint = ScimEndpoint;
