"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserEndpoint = void 0;
const accessors_1 = require("@rocket.chat/apps-engine/definition/accessors");
const Error_1 = require("../scim/Error");
const User_1 = require("../scim/User");
const user_1 = require("./common/user");
const ScimEndpoint_1 = require("./ScimEndpoint");
class UserEndpoint extends ScimEndpoint_1.ScimEndpoint {
    constructor() {
        super(...arguments);
        this.path = "Users/:id";
    }
    async _get(ctx) {
        const o = await ctx.rc.user.info(ctx.id());
        this.handleError(o);
        const user = User_1.SCIMUser.fromRC(o.user);
        const m = await ctx.store.getUser(ctx.id());
        if (m) {
            user.externalId = m.externalId;
        }
        return this.success(user);
    }
    async _put(ctx) {
        const u = User_1.SCIMUser.fromPlain(ctx.content());
        if (ctx.id() === ctx.rc.userId) {
            u.active = true;
            if (!u.roles.find((x) => x.value === "admin")) {
                u.roles.push({ value: "admin" });
            }
        }
        const roles = await (0, user_1.findOrCreateRole)(ctx, u);
        const o = await ctx.rc.user.update({
            userId: ctx.id(),
            data: {
                email: u.getEmail(),
                name: u.displayName || undefined,
                username: u.userName,
                active: u.active,
                verified: true,
                roles,
                customFields: {
                    scimExternalId: u.externalId,
                },
            },
        });
        this.handleError(o);
        await ctx.store.saveUser(ctx.id(), { externalId: u.externalId });
        const user = User_1.SCIMUser.fromRC(o.user);
        user.externalId = u.externalId;
        return this.success(user);
    }
    async _delete(ctx) {
        if (ctx.id() === ctx.rc.userId) {
            throw new Error_1.SCIMError()
                .setStatus(accessors_1.HttpStatusCode.FORBIDDEN)
                .setScimType(Error_1.SCIMErrorType.MUTABILITY)
                .setDetail("The admin user can't be deleted");
        }
        const o = await ctx.rc.user.delete({
            userId: ctx.id(),
            confirmRelinquish: true,
        });
        this.handleError(o);
        await ctx.store.deleteUser(ctx.id());
        return this.response({
            status: accessors_1.HttpStatusCode.NO_CONTENT,
        });
    }
}
exports.UserEndpoint = UserEndpoint;
