"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RcSdk = void 0;
const EmptyResponseError_1 = require("../errors/EmptyResponseError");
const JsonParseError_1 = require("../errors/JsonParseError");
const RcSdkRole_1 = require("./RcSdkRole");
const RcSdkTeam_1 = require("./RcSdkTeam");
const RcSdkUser_1 = require("./RcSdkUser");
class RcSdk {
    constructor(username, password, http, logger) {
        this.baseUrl = "http://localhost:3000/api/v1";
        this.userId = username;
        this.headers = {
            "X-User-Id": username,
            "X-Auth-Token": password,
            "Content-Type": "application/json",
        };
        this.http = http;
        this.logger = logger;
        this.user = new RcSdkUser_1.RcSdkUser(this);
        this.team = new RcSdkTeam_1.RcSdkTeam(this);
        this.role = new RcSdkRole_1.RcSdkRole(this);
    }
    get(url, content) {
        return this.request(this.http.get, url, content);
    }
    post(url, content) {
        return this.request(this.http.post, url, content);
    }
    put(url, content) {
        return this.request(this.http.put, url, content);
    }
    del(url, content) {
        return this.request(this.http.del, url, content);
    }
    patch(url, content) {
        return this.request(this.http.patch, url, content);
    }
    parseResponse(response) {
        if (!response.content) {
            throw new EmptyResponseError_1.EmptyResponseError();
        }
        let content;
        try {
            content = JSON.parse(response.content);
        }
        catch (e) {
            throw new JsonParseError_1.JsonParseError();
        }
        return content;
    }
    buildUrl(url) {
        return `${this.baseUrl}/${url}`;
    }
    async buildOptions(content) {
        const options = {
            headers: this.headers,
        };
        if (content !== undefined) {
            options.content = JSON.stringify(content);
        }
        return options;
    }
    async request(method, url, content) {
        this.logger.debug(`API Request ${url}`, content);
        const response = await method.bind(this.http)(this.buildUrl(url), await this.buildOptions(content));
        const o = this.parseResponse(response);
        this.logger.debug(`API Response`, o);
        return o;
    }
}
exports.RcSdk = RcSdk;
