<?php

declare(strict_types=1);

namespace OCA\SCIMServiceProvider\Controller;

use OCA\SCIMServiceProvider\Responses\SCIMJSONResponse;
use OCA\SCIMServiceProvider\Responses\SCIMListResponse;
use OCA\SCIMServiceProvider\Service\UserService;
use OCA\SCIMServiceProvider\Util\Util;
use OCP\AppFramework\ApiController;
use OCP\AppFramework\Http\Response;
use OCP\IRequest;
use Opf\Models\SCIM\Standard\Users\CoreUser;

class UserController extends ApiController {
	private UserService $userService;

	public function __construct(
		string $appName,
		IRequest $request,
		UserService $userService
	) {
		parent::__construct($appName, $request);

		$this->userService = $userService;
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @return SCIMListResponse
	 */
	public function index(): SCIMJSONResponse {
		$params = $this->request->getParams();
		$params = Util::parsePagination($params);
		$scimUsers = $this->userService->getAll($params["startIndex"], $params["count"]);
		$list = [];
		foreach ($scimUsers as $scimUser) {
			$list[] = $scimUser->toSCIM(false, Util::getBaseUrl($this->request));
		}
		$total = $this->userService->countAll();
		return new SCIMListResponse($list, $params["startIndex"], $total);
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $id
	 * @return SCIMJSONResponse
	 */
	// TODO: Add filtering support here as well
	public function show(string $id): SCIMJSONResponse {
		$scimUser = $this->userService->get($id);
		return new SCIMJSONResponse($scimUser->toSCIM(false, Util::getBaseUrl($this->request)));
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $displayName
	 * @param array  $members
	 * @return SCIMJSONResponse
	 */
	public function create(): SCIMJSONResponse {
		$params = $this->request->getParams();

		$scimUser = new CoreUser();
		$scimUser->fromSCIM($params);

		$scimGroup = $this->userService->create($scimUser);
		return new SCIMJSONResponse($scimGroup->toSCIM(false, Util::getBaseUrl($this->request)), 201);
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $id
	 * @return SCIMJSONResponse
	 */
	public function update(string $id): SCIMJSONResponse {
		$params = $this->request->getParams();

		$scimUser = new CoreUser();
		$scimUser->fromSCIM($params);

		$scimUser = $this->userService->update($id, $scimUser);
		return new SCIMJSONResponse($scimUser->toSCIM(false, Util::getBaseUrl($this->request)));
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $id
	 * @return Response
	 */
	public function destroy(string $id): Response {
		$this->userService->destroy($id);

		$response = new Response();
		$response->setStatus(204);
		return $response;
	}
}
