<?php

declare(strict_types=1);

namespace OCA\SCIMServiceProvider\Service;

use Exception;
use OCP\IUser;
use OCP\IUserManager;
use OCP\Security\ISecureRandom;
use Opf\Models\SCIM\Standard\Meta;
use Opf\Models\SCIM\Standard\MultiValuedAttribute;
use Opf\Models\SCIM\Standard\Users\CoreUser;
use Psr\Log\LoggerInterface;

class UserService {
	/** @var LoggerInterface */
	private $logger;

	private IUserManager $userManager;

	/** @var \OCP\Security\ISecureRandom */
	private $secureRandom;

	public function __construct(LoggerInterface $logger, IUserManager $userManager, ISecureRandom $secureRandom) {
		$this->logger = $logger;
		$this->userManager = $userManager;
		$this->secureRandom = $secureRandom;
	}

	private function toSCIM(IUser $ncUser): CoreUser {
		$scimUser = new CoreUser();
		$scimUser->setId($ncUser->getUID());
		$scimUser->setUserName($ncUser->getUID());
		$scimUser->setDisplayName($ncUser->getDisplayName());
		$scimUser->setActive($ncUser->isEnabled());

		$emails = [];
		if (!empty($ncUser->getSystemEMailAddress())) {
			$email = new MultiValuedAttribute();
			$email->setValue($ncUser->getSystemEMailAddress());
			$emails[] = $email;
		}
		$scimUser->setEmails($emails);

		$meta = new Meta();
		$meta->setResourceType("User");

		$scimUser->setMeta($meta);

		return $scimUser;
	}


	private function updateUser(IUser $ncUser, CoreUser $scimUser) {
		$displayName = $scimUser->getDisplayName();
		if (isset($displayName)) {
			$ncUser->setDisplayName($displayName);
		}

		$emails = $scimUser->getEmails();
		if (isset($emails)) {
			if (sizeof($emails) > 0) {
				$ncUser->setSystemEMailAddress($emails[0]->getValue());
			} else {
				$ncUser->setSystemEMailAddress("");
			}
		}

		$active = $scimUser->getActive();
		if ($active !== null) {
			$ncUser->setEnabled($active);
		}
	}

	public function countAll(): int {
		$count = 0;
		foreach ($this->userManager->countUsers() as $key => $value) {
			$count = $count + $value;
		}
		return $count;
	}

	public function getAll(int $startIndex, int $count = null): array {
		$this->logger->info("Reading all users");

		if ($count === 0) {
			return [];
		}

		$ncUsers = $this->userManager->search("", $count, $startIndex - 1);

		$scimUsers = [];
		foreach ($ncUsers as $ncUser) {
			$scimUsers[] = $this->toSCIM($ncUser);
		}

		return $scimUsers;
	}


	public function get(string $id): CoreUser {
		$this->logger->info("Reading user with ID: " . $id);

		if (!$this->userManager->userExists($id)) {
			throw new Exception("Not found", 404);
		}

		$ncUser = $this->userManager->get($id);
		return $this->toSCIM($ncUser);
	}

	public function create(CoreUser $scimUser): CoreUser {
		$this->logger->info("Creating user with userName: " . $scimUser->getUserName());


		if ($this->userManager->userExists($scimUser->getUserName())) {
			$this->logger->error("User to be created already exists");
			throw new Exception('User exists', 409);
		}

		$ncUser = $this->userManager->createUser($scimUser->getUserName(), $this->secureRandom->generate(64));
		$this->updateUser($ncUser, $scimUser);

		return $this->get($ncUser->getUID());
	}

	public function update(string $id, CoreUser $scimUser): CoreUser {
		$this->logger->info("Updating user with ID: " . $id);

		if (!$this->userManager->userExists($id)) {
			throw new Exception("Not found", 404);
		}

		$this->logger->info($scimUser->toSCIM());

		$ncUser = $this->userManager->get($id);
		$this->updateUser($ncUser, $scimUser);

		return $this->get($id);
	}

	public function destroy(string $id): void {
		$this->logger->info("Deleting user with ID: " . $id);

		if (!$this->userManager->userExists($id)) {
			throw new Exception('User not found', 404);
		}
		$ncUser = $this->userManager->get($id);
		$ncUser->delete();
	}
}
