<?php

declare(strict_types=1);

namespace OCA\SCIMServiceProvider\Controller;

use OCA\SCIMServiceProvider\Responses\SCIMJSONResponse;
use OCA\SCIMServiceProvider\Responses\SCIMListResponse;
use OCA\SCIMServiceProvider\Service\GroupService;
use OCA\SCIMServiceProvider\Util\Util;
use OCP\AppFramework\ApiController;
use OCP\AppFramework\Http\Response;
use OCP\IRequest;
use Opf\Models\SCIM\Standard\Groups\CoreGroup;

class GroupController extends ApiController {
	private GroupService $groupService;

	public function __construct(
		string $appName,
		IRequest $request,
		GroupService $groupService
	) {
		parent::__construct($appName, $request);

		$this->groupService = $groupService;
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @return SCIMListResponse
	 */
	public function index(): SCIMJSONResponse {
		$params = $this->request->getParams();
		$params = Util::parsePagination($params);
		$scimGroups = $this->groupService->getAll($params["startIndex"], $params["count"]);
		$list = [];
		foreach ($scimGroups as $scimGroup) {
			$list[] = $scimGroup->toSCIM(false, Util::getBaseUrl($this->request));
		}
		$total = $this->groupService->countAll();
		return new SCIMListResponse($list, $params["startIndex"], $total);
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $id
	 * @return SCIMJSONResponse
	 */
	// TODO: Add filtering support here as well
	public function show(string $id): SCIMJSONResponse {
		$scimGroup = $this->groupService->get($id);
		return new SCIMJSONResponse($scimGroup->toSCIM(false, Util::getBaseUrl($this->request)));
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $displayName
	 * @param array  $members
	 * @return SCIMJSONResponse
	 */
	// public function create(string $displayName, array $members = []): SCIMJSONResponse
	public function create(): SCIMJSONResponse {
		$params = $this->request->getParams();

		$scimGroup = new CoreGroup();
		$scimGroup->fromSCIM($params);

		$scimGroup = $this->groupService->create($scimGroup);
		return new SCIMJSONResponse($scimGroup->toSCIM(false, Util::getBaseUrl($this->request)), 201);
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $id
	 * @return SCIMJSONResponse
	 */
	public function update(string $id): SCIMJSONResponse {
		$params = $this->request->getParams();

		$scimGroup = new CoreGroup();
		$scimGroup->fromSCIM($params);

		$scimGroup = $this->groupService->update($id, $scimGroup);
		return new SCIMJSONResponse($scimGroup->toSCIM(false, Util::getBaseUrl($this->request)));
	}

	/**
	 * @NoCSRFRequired
	 * @PublicPage
	 *
	 * @param string $id
	 * @return Response
	 */
	public function destroy(string $id): Response {
		$this->groupService->destroy($id);

		$response = new Response();
		$response->setStatus(204);
		return $response;
	}
}
